<?php
/**
 * @package    solo
 * @copyright  Copyright (c)2014-2019 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license    GNU GPL version 3 or later
 */

namespace Solo\View\Dbfilters;

use Awf\Html\Select;
use Awf\Text\Text;
use Awf\Uri\Uri;
use Awf\Utils\Template;
use Solo\Helper\Escape;
use Solo\View\ViewTraits\ProfileIdAndName;

class Html extends \Solo\View\Html
{
	use ProfileIdAndName;

	/**
	 * SELECT element for choosing a database root
	 *
	 * @var  string
	 */
	public $root_select = '';

	/**
	 * List of database roots
	 *
	 * @var  array
	 */
	public $roots = [];

	/**
	 * The view's interface data encoded in JSON format
	 *
	 * @var  string
	 */
	public $json = '';

	/**
	 * Prepare the view data for the main task
	 *
	 * @return  boolean
	 */
	public function onBeforeMain()
	{
		$model = $this->getModel();
		$task  = $model->getState('browse_task', 'normal', 'cmd');

		$router = $this->container->router;

		// Add custom submenus
		$toolbar = $this->container->application->getDocument()->getToolbar();
		$toolbar->addSubmenuFromDefinition(array(
			'name'  => 'normal',
			'title' => Text::_('COM_AKEEBA_FILEFILTERS_LABEL_NORMALVIEW'),
			'url'   => $router->route('index.php?view=dbfilters&task=normal'),
		));
		$toolbar->addSubmenuFromDefinition(array(
			'name'  => 'tabular',
			'title' => Text::_('COM_AKEEBA_FILEFILTERS_LABEL_TABULARVIEW'),
			'url'   => $router->route('index.php?view=dbfilters&task=tabular'),
		));

		// Get a JSON representation of the available roots
		/** @var \Solo\Model\Dbfilters $model */
		$model     = $this->getModel();
		$root_info = $model->get_roots();
		$roots     = array();
		$options   = array();

		if (!empty($root_info))
		{
			// Loop all db definitions
			foreach ($root_info as $def)
			{
				$roots[]   = $def->value;
				$options[] = Select::option($def->value, $def->text);
			}
		}

		$site_root         = $roots[0];
		$attributes        = 'onchange="akeeba.Dbfilters.activeRootChanged();"';
		$this->root_select = Select::genericList($options, 'root', $attributes, 'value', 'text', $site_root, 'active_root');
		$this->roots       = $roots;

		switch ($task)
		{
			case 'normal':
			default:
				$this->setLayout('default');

				// Get a JSON representation of the directory data
				$model      = $this->getModel();
				$json       = json_encode($model->make_listing($site_root));
				$this->json = $json;
				break;

			case 'tabular':
				$this->setLayout('tabular');

				// Get a JSON representation of the tabular filter data
				$model      = $this->getModel();
				$json       = json_encode($model->get_filters($site_root));
				$this->json = $json;

				break;
		}

		// Load additional Javascript
		Template::addJs('media://js/solo/fsfilters.js', $this->container->application);
		Template::addJs('media://js/solo/dbfilters.js', $this->container->application);

		// Load the Javascript language strings
		$this->loadCommonJavascript();

		$this->getProfileIdAndName();

		return true;
	}

	/**
	 * The normal task simply calls the method for the main task
	 *
	 * @return  boolean
	 */
	public function onBeforeNormal()
	{
		return $this->onBeforeMain();
	}

	/**
	 * The tabular task simply calls the method for the main task
	 *
	 * @return  boolean
	 */
	public function onBeforeTabular()
	{
		return $this->onBeforeMain();
	}

	/**
	 * Load the common Javascript for this feature: language strings, image locations
	 */
	protected function loadCommonJavascript()
	{
		$strings                                               = array();
		$strings['COM_AKEEBA_FILEFILTERS_LABEL_UIROOT']        = Escape::escapeJS(Text::_('COM_AKEEBA_FILEFILTERS_LABEL_UIROOT'));
		$strings['COM_AKEEBA_FILEFILTERS_LABEL_UIERRORFILTER'] = Escape::escapeJS(Text::_('COM_AKEEBA_FILEFILTERS_LABEL_UIERRORFILTER'));
		$strings['COM_AKEEBA_DBFILTER_TABLE_META_ROWCOUNT']    = Escape::escapeJS(Text::_('COM_AKEEBA_DBFILTER_TABLE_META_ROWCOUNT'));

		$filters = array('tables', 'tabledata');
		foreach ($filters as $type)
		{
			$key           = 'COM_AKEEBA_DBFILTER_TYPE_' . strtoupper($type);
			$strings[$key] = Escape::escapeJS(Text::_($key));
		}

		$types = array('misc', 'table', 'view', 'procedure', 'function', 'trigger');
		foreach ($types as $type)
		{
			$key           = 'COM_AKEEBA_DBFILTER_TABLE_' . strtoupper($type);
			$strings[$key] = Escape::escapeJS(Text::_($key));
		}

		$loadingGif = Escape::escapeJS(Uri::base(false, $this->container) . 'media/loading.gif');
		$ajaxURL    = Escape::escapeJS($this->getContainer()->router->route('index.php?view=dbfilters&task=ajax&format=raw'));
		$json       = Escape::escapeJS($this->json, "'");

		$js = <<< JS
		
var akeeba_dbfilter_data = eval($json);

akeeba.loadScripts.push(function() {
		akeeba.Dbfilters.translations['COM_AKEEBA_FILEFILTERS_LABEL_UIROOT'] = '{$strings['COM_AKEEBA_FILEFILTERS_LABEL_UIROOT']}';
		akeeba.Dbfilters.translations['COM_AKEEBA_FILEFILTERS_LABEL_UIERRORFILTER'] = '{$strings['COM_AKEEBA_FILEFILTERS_LABEL_UIERRORFILTER']}';
		akeeba.Fsfilters.translations['COM_AKEEBA_FILEFILTERS_LABEL_UIERRORFILTER'] = '{$strings['COM_AKEEBA_FILEFILTERS_LABEL_UIERRORFILTER']}';

		akeeba.Dbfilters.translations['COM_AKEEBA_DBFILTER_TYPE_TABLES'] = '{$strings['COM_AKEEBA_DBFILTER_TYPE_TABLES']}';
		akeeba.Dbfilters.translations['COM_AKEEBA_DBFILTER_TYPE_TABLEDATA'] = '{$strings['COM_AKEEBA_DBFILTER_TYPE_TABLEDATA']}';

		akeeba.Dbfilters.translations['COM_AKEEBA_DBFILTER_TABLE_MISC'] = '{$strings['COM_AKEEBA_DBFILTER_TABLE_MISC']}';
		akeeba.Dbfilters.translations['COM_AKEEBA_DBFILTER_TABLE_TABLE'] = '{$strings['COM_AKEEBA_DBFILTER_TABLE_TABLE']}';
		akeeba.Dbfilters.translations['COM_AKEEBA_DBFILTER_TABLE_VIEW'] = '{$strings['COM_AKEEBA_DBFILTER_TABLE_VIEW']}';
		akeeba.Dbfilters.translations['COM_AKEEBA_DBFILTER_TABLE_PROCEDURE'] = '{$strings['COM_AKEEBA_DBFILTER_TABLE_PROCEDURE']}';
		akeeba.Dbfilters.translations['COM_AKEEBA_DBFILTER_TABLE_FUNCTION'] = '{$strings['COM_AKEEBA_DBFILTER_TABLE_FUNCTION']}';
		akeeba.Dbfilters.translations['COM_AKEEBA_DBFILTER_TABLE_TRIGGER'] = '{$strings['COM_AKEEBA_DBFILTER_TABLE_TRIGGER']}';
		akeeba.Dbfilters.translations['COM_AKEEBA_DBFILTER_TABLE_META_ROWCOUNT'] = '{$strings['COM_AKEEBA_DBFILTER_TABLE_META_ROWCOUNT']}';

		akeeba.Dbfilters.loadingGif = '$loadingGif';
		akeeba.System.params.AjaxURL = '$ajaxURL';
});

JS;

		$this->getContainer()->application->getDocument()->addScriptDeclaration($js);
	}

}
