<?php
/*
 * 	BNE CMB2 Render Custom Field Type Class
 *
 *	This class allows to easliy add a new field type to CMB2. It expects
 *	the field type to a multiple group -- more than one individual field.
 *
 * 	@author  	Kerry Kline, BNE Creative
 * 	@link    	http://www.bnecreative.com
 * 	@version 	1.2.3
 * 	@updated 	Aug 15, 2017
 * 	License: 	GPL2

    Copyright 2017  BNE Creative

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License version 2,
    as published by the Free Software Foundation.

    You may NOT assume that you can use any other version of the GPL.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    The license for this software can likely be found here:
    http://www.gnu.org/licenses/gpl-2.0.html

*/

// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;


/* EXAMPLE
$field = new BNE_CMB_Render_Field(
	// Field Type ID
	'my_custom_field_type',
	
	// Fields (Requires 2 or more fields to render)								
	array(

		// Field #1
		array(
			// Field Wrapper Args
			'label'				=>	'Text',				// Label shown above the field
			'column'			=>	'12',				// Grid: 1, 1/2, 1/3, 1/4
			'label_location'	=>	'above',			// Field Label: above | below
			
			// Field Args
			'field'			=>	array(
				'id'			=>	'example_text',		// Unique Field ID
				'type'			=>	'text',				// Field Type: text | textarea | select | radio | checkbox | colorpicker
				'input'			=>	'url',				// Text Only Input: text | url | tel | number
				'default'		=>	'',					// Default value if empty
				'desc'			=>	'XXX'				// Checkbox only label
			),

			// Field Attributes
			'field_atts'	=>	array(
				'rows'			=>	'4',				// Textarea only
				'required'		=>	'required',			// Set the field to being required
				'placeholder'	=>	'XXX'				// Add a placeholder to the field
			),
			
			// Field Options for Select/Radio
			'field_options'	=>	array(
				''				=>	'Select',			// Empty select option, Needed if is repeatable
				'male'			=>	'Male',				// value : Name
				'female'		=>	'Female',			// Etc...
			)

		) // End Field

	) // End Field(s)

); // End $field
*/



if( !class_exists( 'BNE_CMB_Render_Field' ) ) {

	class BNE_CMB_Render_Field {


	    /*
	     * 	Constructor
	     *
	     *	$type	(string)	The field ID from CMB2
	     *	@fields	(array)		An array of fields to create within a group
	     *
	    */
		function __construct( $type, $fields = array() ) {
			
			// Assignments
			$this->type = $type;
			$this->fields = $fields;
			$this->multiple = ( count( $this->fields ) > 1 ) ? true : false;
			
			// Stop if the field type does not have multiple values!
			if( $this->multiple == false ) {
				return;
			}
						
			// Register New Field
			add_action( "cmb2_render_{$type}", array( $this, 'render_field' ), 10, 5 );
			
			// Sanitize and Escape Repeated fields
			add_filter( "cmb2_sanitize_{$type}", array( $this, 'sanitize_field' ), 10, 5 );
			add_filter( "cmb2_types_esc_{$type}", array( $this, 'escape_field' ), 10, 4 );
			
		}


	    /*
	     * 	Render New Field into CMB2
	     *	
	    */
		function render_field( $cmb_field, $escaped_value, $object_id, $object_type, $field_type_object ) {
			
			// Field Properties
			$cmb_field = $this->type;
			$fields = $this->fields;
			//$value = $escaped_value;
						
			// Field Description
			if( $field_type_object->_desc( true ) ) {
				echo '<div class="bne-field-description">'.$field_type_object->_desc().'</div>';
			}

			// Open a new field Row
			echo '<div class="bne-cmb-field-group">';
				
				// Start a field count
				$i = 1;
				$total = count($fields);
				
				
				// Loop through each new field
				foreach( $fields as $field ) {
					
					// Defaults
					$field = wp_parse_args( $field, array(
						'column'			=>	'12',
						'label'				=>	'',
						'label_location'	=>	'above',
						'field_atts'		=>	'',
						'field_options'		=>	'',
						'value'				=>	'',
					) );
					
															
					// Field Type
					$field_type = $field['field'];
					
					// Field Attributes
					$field_atts = (array) $field['field_atts'];

					// Field Options
					$field_options = (array) $field['field_options'];

					// Field ID's & Value
					$id = '_'.$field_type['id'];
					$name = '['.$field_type['id'].']';
					
					// Set Value
					if( isset( $escaped_value[$field_type['id']] ) ) {
						$field['value'] = $escaped_value[$field_type['id']];
					}
					
					// Place default as value if there is no value
					if( !$field['value'] && isset( $field_type['default'] ) ) {
						$field['value'] = $field_type['default'];
					}

					// Required Field Args 
					$field_args = array(
						'id'			=>	$field_type_object->_id( $id ),
						'name'			=>	$field_type_object->_name( $name ),
						'desc'			=>	'',
						'type'			=>	$field_type['type'],
						'value' 		=> 	$field['value'],
					);
					
					// Merge the atts into the field arguemnts
					if( $field_atts ) {
						$field_args = array_merge( $field_args, $field_atts );
					}
					
					// Begin Field
					echo '<div class="bne-cmb-field bne-cmb-col-'.str_replace('/', '-', $field['column']).' field'.$id.'">';
						

						// Field Label
						if( 'above' == $field['label_location'] && $field['label'] ) {
							echo '<label class="field_label">'.$field['label'].'</label>';
						}
						

						// Text
						if( 'text' == $field_type['type'] ) {
							// Adjust input type ( text, email, tel, url, etc)
							if( isset( $field_type['input'] ) ) {
								$field_args['type'] = $field_type['input'];
							}
							echo $field_type_object->input( $field_args );
						}
						

						// Textarea
						if( 'textarea' == $field_type['type'] ) {
							echo $field_type_object->textarea( $field_args );
						}
						

						// Select
						if( 'select' == $field_type['type'] ) {
							$options = '';
							if( $field_options ) {
								foreach( $field_options as $k => $v ) {
									$options .= '<option value="'.$k.'" '.selected( $field_args['value'], $k, false ).'>'.$v.'</option>';
								}
							}
							$field_args['options'] = $options;
							echo $field_type_object->select( $field_args );
							
							/*
							$field_type_object->type = new CMB2_Type_Select( $field_type_object );
							$options = '';
							foreach( $field_options as $option_value => $option_label ) {
								$args = array(
									'value' => $option_value,
									'label' => $option_label,
								);
								$args['checked'] = ( $field_args['value'] == $option_value ) ? 'checked' : '';
								$options .= $field_type_object->select_option( $args );
							}

							$field_args['options'] = $options;
							echo $field_type_object->select( $field_args );
							*/
							
						}

						
						// Radio
						if( 'radio' == $field_type['type'] ) {
							$options = '';
							if( $field_options ) {
								foreach( $field_options as $k => $v ) {
									$options .= '<li>';
									$options .= '<input type="radio" value="'.$k.'" '.checked( $field_args['value'], $k, false ).' name="'.$field_args['name'].'" id="'.$field_args['id'].'" class="cmb2-option">';
									$options .= '<label for="'.$field_args['id'].'">'.$v.'</label>';
									$options .= '</li>';
								}
							}
							$field_args['options'] = $options;
							echo $field_type_object->radio( $field_args );
						}

	
						// Checkbox
						if( 'checkbox' == $field_type['type'] ) {
							$desc = ( isset( $field_type['desc'] ) ) ? $field_type['desc'] : $label;
							echo '<input type="checkbox" class="cmb2-option cmb2-list" name="'.$field_args['name'].'" id="'.$field_args['id'].'" value="on" '.checked( $field_args['value'], 'on', false ).'>';
							echo '<label for="'.$field_args['id'].'">'.$desc.'</label>';
						}


						// Colorpicker
						if( 'colorpicker' == $field_type['type'] ) {
							unset( $field_args['type'] );
							echo $field_type_object->colorpicker( $field_args, '#' );
						}


						// Field Label
						if( 'below' == $field['label_location'] && $field['label'] ) {
							echo '<label class="field_label label-below">'.$field['label'].'</label>';
						}

						
					// End Field
					echo '</div>'; 
					
					
				} // End $this->fields loop

			echo '</div><!-- .bne-cmb-row (end) -->';

		} // End render_field


	    /*
	     * 	Repeater Sanitize
	     *	
	    */
		function sanitize_field( $check, $meta_value, $object_id, $field_args, $sanitize_object ) {
			// if not repeatable, bail out.
			if ( ! is_array( $meta_value ) || ! $field_args['repeatable'] ) {
				return $check;
			}
			
			foreach ( $meta_value as $key => $val ) {
				$meta_value[ $key ] = array_filter( array_map( 'sanitize_text_field', $val ) );
			}
			
			return array_filter( $meta_value );
		}
		
	
	    /*
	     * 	Repeater Escaping
	     *	
	    */
		function escape_field( $check, $meta_value, $field_args, $field_object ) {
			// if not repeatable, bail out.
			if ( ! is_array( $meta_value ) || ! $field_args['repeatable'] ) {
				return $check;
			}
			
			foreach ( $meta_value as $key => $val ) {
				if( $val )
				$meta_value[ $key ] = array_filter( array_map( 'esc_attr', $val ) );
			}
			
			return array_filter( $meta_value );
		}


	} // End class

} // End class_exists