<?php
/*
 * 	Framework Support functions
 *
 * 	@author		Kerry Kline
 * 	@copyright	Copyright (c) Kerry Kline
 * 	@link		http://www.bnecreative.com
 * 	@package 	BNE WordPress Framework
 *
*/

// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;


/*
 *	Dev - A Custom Print out a function/array for testing.
*/
if( !function_exists( 'bne_printr' ) ) {
	function bne_printr( $test = '' ) {
		$style = 'style="margin-left:180px; margin-top: 10px; border: 1px solid #ccc; padding: 10px; background: white; max-width: calc(100vw - 250px); word-break: break-all; word-wrap: break-word;"';
		$margin = (is_admin()) ? $style : '';
		echo '<pre '.$margin.'>';
		echo print_r( $test );
		echo '</pre>';
	}
}



/*
 * 	Filter Dynamic Variables
 *
 *	Certain areas of the framework allows a variable to pass.
 *	This allows replacing those variables on the output.
 *
 * 	@since 		v1.4.2
 *	@udpated	v2.0
 *
*/
function bne_dynamic_variables( $text ) {
	
	// Built in Vars
	$vars = array(
		'{year}'		=>	esc_attr( date('Y') ),
		'{site_title}'	=>	esc_html( get_bloginfo('site_title') ),
	);
	
	// FA Icon
	preg_match_all('/{icon_(.*?)\}/s', $text, $matches );
	foreach( $matches[1] as $icon => $icon_name ) {
		$vars['{icon_'.$icon_name.'}'] = '<i class="bne-fa fa fa-'.$icon_name.'"></i>';
	}
	
	return strtr( $text, $vars );
}



/*
 * 	Converts a multidimensional array of CSS rules into a CSS string.
 *
 *	@url https://matthewgrasmick.com/posts/convert-nested-php-array-css-string
 *
 * 	@param array $rules
 *   	An array of CSS rules in the form of:
 *   	array('selector'=>array('property' => 'value')). Also supports selector
 *   	nesting, e.g.,
 *   	array('selector' => array('selector'=>array('property' => 'value'))).
 *
 * 	@return string	A CSS string of rules. This is not wrapped in <style> tags.
 *
 * 	@since 		v2.2.7
*/
function bne_generate_css_properties( $rules, $indent = 0 ) {
	$css = '';
	$prefix = str_repeat( '  ', $indent );
	
	if( $rules ) {
		foreach( $rules as $key => $value ) {
			
			if( is_array( $value ) ) {
		
				$selector = $key;
				$properties = $value;
		
				$css .= $prefix . "$selector {\n";
				$css .= $prefix .bne_generate_css_properties( $properties, $indent + 1 );
				$css .= $prefix . "}\n";
			
			} else {
				
				// Background Image URL
				if( $key == 'background-image' ) {
					$value = 'url('.$value.')';
				}
				
				$property = $key;
				$css .= $prefix . "$property: $value;\n";
			}
		
		}
				
	}

	return $css;

}




/*
 *	Build an array of Websafe (OS) Fonts.
 *
 * 	@since v1.0
 *
*/
function bne_typography_get_os_fonts() {

    // Websafe Font Defaults
    $os_faces = array(
		'Arial, "Helvetica Neue", Helvetica, sans-serif' => 'Arial',
		'Baskerville, "Baskerville Old Face", "Hoefler Text", Garamond, "Times New Roman", serif' => 'Baskerville',
		'"Century Gothic", CenturyGothic, AppleGothic, sans-serif' => 'Century Gothic',
		'Georgia, Times, "Times New Roman", serif' => 'Georgia',
		'"Helvetica Neue", Helvetica, Arial, sans-serif' => 'Helvetica',
		'Impact, Charcoal, "Arial Black", sans serif;' => 'Impact',
		'"Lucida Grande", "Lucida Sans Unicode", "Lucida Sans", Geneva, Verdana, sans-serif' => 'Lucida Grande',
		'Palatino, "Palatino Linotype", Georgia, serif' => 'Palatino',
		'Tahoma, Geneva, Verdana, sans-serif' => 'Tahoma',
		'TimesNewRoman, "Times New Roman", Times, Baskerville, Georgia, serif' => 'Times New Roman',
		'"Trebuchet MS", "Lucida Grande", "Lucida Sans Unicode", "Lucida Sans", Tahoma, sans-serif' => 'Trebuchet',
		'Verdana, Geneva, sans-serif' => 'Verdana',
    );

    return $os_faces;
}



/*
 *	Google Font List Array
 *
 *	@note:	apply_filters( 'bne_google_font_api_key', '') allows a api key
 *	@note:	if no API key is entered, a fallback list from
 *			/framework/assets/fonts/google-fonts.php will be used.
 *
 * 	@since 		v1.0
 *	@updated 	v2.0
 *
*/
function bne_typography_get_google_fonts() {

	// Open Goole Font Array
	$google_faces = get_transient( 'bne_'.get_template().'_google_fonts' );

	// Check if Google Fonts are supported within the framework
	if( !$google_faces ) {
		
		// Get the Google API Key
		$google_api_key = apply_filters( 'bne_google_font_api_key', '');

		// Google API
		if( $google_api_key ) {

			$google_api_url = esc_url( 'https://www.googleapis.com/webfonts/v1/webfonts?key='.$google_api_key );
			$response = wp_remote_retrieve_body( wp_remote_get( $google_api_url, array('sslverify' => false ) ) );

		// Google API not available, use fallback font list.
		} else {
			include_once( BNE_FRAMEWORK_DIR . '/assets/fonts/google-fonts.php' );
			$response = $google_fallback_fonts;
		}

		// Decode the retreived json file
		$json_fonts = json_decode( $response, true );
		$google_fonts = $json_fonts['items'];

		// Loop through the Google Font stacks and pull out the family names.
		if( $google_fonts ) {

			foreach( $google_fonts as $google_font ) {
				$google_faces[$google_font['family']] = $google_font['family'];
			}
			
			// Cache result
			$time = 86400 * 7;
			set_transient( 'bne_'.get_template().'_google_fonts', $google_faces, $time ); // 7 days

		}
		
	}

	return $google_faces;
}



/*
 * 	Enqueue a Google font.
 *
 *	Fonts area added using bne_enqueue_google_font() and returning
 *	a font name along with their variants.
 *
 *	ex: $font[] = 'Dosis:400,600,700';
 *
 * 	@since 		v1.0
 * 	@udpated 	v2.0
 *
*/
if( !function_exists( 'bne_typography_enqueue_google_font' ) ) :
	function bne_typography_enqueue_google_font() {

		// Gets Google Font Array
		//$google_fonts = array_keys( bne_typography_get_google_fonts() );
		$enqueue_fonts = apply_filters( 'bne_enqueue_google_font', array() );
		
		// merge fonts into a single request
		$query_args = array(
		   'family' => implode('|', $enqueue_fonts),
		   'display' => 'swap'
		);
		
		if( !empty( $enqueue_fonts ) ) {
			wp_enqueue_style( 
				'google-fonts', 
				add_query_arg( $query_args, '//fonts.googleapis.com/css' ), 
				array(), 
				null 
			);
		}	 	
	}
	add_action( 'wp_enqueue_scripts', 'bne_typography_enqueue_google_font' );
endif;



/*
 *	Get recognized font faces
 *
 *	An array of font faces used in Theme Options.
 *	It merges bne_typography_get_os_fonts and bne_typography_get_google_fonts
 *
 * 	@since	v1.4.2
*/
function bne_recognized_font_faces() {	
	$faces = array_merge( bne_typography_get_os_fonts() , bne_typography_get_google_fonts() );	
	return $faces;
}



/*
 *	Get Recognized Font Styles
 *
 *	An array of font styles used in Theme Options.
 *
 * 	@since	v1.4.2
*/
function bne_recognized_font_styles( $default ) {
	$default = array(
		'normal' 			=> 	esc_html__( 'Normal', 'bne' ),
		'uppercase' 		=> 	esc_html__( 'Uppercase', 'bne' ),
		'italic' 			=> 	esc_html__( 'Italic', 'bne' ),
		'uppercase-italic' 	=> 	esc_html__( 'Uppercase Italic', 'bne' )
	);
	return apply_filters( 'bne_recognized_font_styles', $default );
}



/*
 *	Get Recognized Font Weights
 *
 *	An array of font weights used in Theme Options.
 *
 * 	@since	v1.4.2
*/
function bne_recognized_font_weights() {
	$default = array(
		'100'	=>	esc_html__( '100', 'bne' ),
		'200'	=>	esc_html__( '200', 'bne' ),
		'300'	=>	esc_html__( '300', 'bne' ),
		'400'	=>	esc_html__( '400 (normal)', 'bne' ),
		'500'	=>	esc_html__( '500', 'bne' ),
		'600'	=>	esc_html__( '600', 'bne' ),
		'700'	=>	esc_html__( '700 (bold)', 'bne' ),
		'800'	=>	esc_html__( '800', 'bne' ),
		'900'	=>	esc_html__( '900', 'bne' )
	);
	return apply_filters( 'bne_recognized_font_weights', $default );
}



/*
 *	Get recognized Background Repeats
 *
 *	An array of background repeat used in Theme Options and CMB2.
 *
 * 	@since	v1.4.2
*/
function bne_recognized_background_repeats() {
	$default = array(
		'no-repeat' =>	esc_html__( 'Background Repeat: No Repeat', 'bne' ),
		'repeat-x' 	=> 	esc_html__( 'Background Repeat: Repeat Horizontally', 'bne' ),
		'repeat-y' 	=> 	esc_html__( 'Background Repeat: Repeat Vertically', 'bne' ),
		'repeat' 	=> 	esc_html__( 'Background Repeat: Repeat All', 'bne' )
	);
	return apply_filters( 'bne_recognized_background_repeats', $default );
}



/*
 *	Get recognized Background Positions
 *
 *	An array of background positions used in Theme Options and CMB2.
 *
 * 	@since	v1.4.2
*/
function bne_recognized_background_positions() {
	$default = array(
		'left top' 		=> 	esc_html__( 'Background Position: Left Top', 'bne' ),
		'left center' 	=> 	esc_html__( 'Background Position: Left Center', 'bne' ),
		'left bottom' 	=> 	esc_html__( 'Background Position: Left Bottom', 'bne' ),
		'right top' 	=> 	esc_html__( 'Background Position: Right Top', 'bne' ),
		'right center' 	=> 	esc_html__( 'Background Position: Right Center', 'bne' ),
		'right bottom' 	=> 	esc_html__( 'Background Position: Right Bottom', 'bne' ),
		'center top' 	=> 	esc_html__( 'Background Position: Center Top', 'bne' ),
		'center center' => 	esc_html__( 'Background Position: Center Center', 'bne' ),
		'center bottom' => 	esc_html__( 'Background Position: Center Bottom', 'bne' ),
	);
	return apply_filters( 'bne_recognized_background_positions', $default );
}



/*
 *	Get recognized Background Sizes
 *
 *	An array of background sizes used in Theme Options and CMB2.
 *
 * 	@since	v1.4.2
*/
function bne_recognized_background_sizes() {
	$default = array(
		'auto' 			=> 	esc_html__( 'Background Size: Auto', 'bne' ),
		'cover' 		=> 	esc_html__( 'Background Size: Cover', 'bne' ),
		'contain' 		=> 	esc_html__( 'Background Size: Contain', 'bne' ),
		'100% 100%' 	=> 	esc_html__( 'Background Size: 100% x 100%', 'bne' ),
		'100% auto' 	=> 	esc_html__( 'Background Size: Fit Horizontally', 'bne' ),
		'auto 100%' 	=> 	esc_html__( 'Background Size: Fit Vertically', 'bne' )
	);
	return apply_filters( 'bne_recognized_background_sizes', $default );
}



/*
 *	Get recognized Background Sizes
 *
 *	An array of background sizes used in Theme Options and CMB2.
 *
 * 	@since	v1.4.2
*/
function bne_recognized_background_attachments() {
	$default = array(
		'scroll' 	=> 	esc_html__( 'Background Scrolling: Normal', 'bne' ),
		'fixed' 	=> 	esc_html__( 'Background Scrolling: Fixed in Place', 'bne' )
	);
	return apply_filters( 'bne_recognized_background_attachments', $default );
}



/*
 *	Get recognized Shade Opacity
 *
 *	An array of opacity settings used in Theme Options and CMB2.
 *
 * 	@since	v1.4.2
*/
function bne_recognized_background_shade_opacity() {
	$default = array(
		'0.05'	=>	esc_html__( 'Shade Opacity: 5%', 'bne' ),
		'0.1'	=>	esc_html__( 'Shade Opacity: 10%', 'bne' ),
		'0.15'	=>	esc_html__( 'Shade Opacity: 15%', 'bne' ),
		'0.2'	=>	esc_html__( 'Shade Opacity: 20%', 'bne' ),
		'0.25'	=>	esc_html__( 'Shade Opacity: 25%', 'bne' ),
		'0.3'	=>	esc_html__( 'Shade Opacity: 30%', 'bne' ),
		'0.35'	=>	esc_html__( 'Shade Opacity: 35%', 'bne' ),
		'0.4'	=>	esc_html__( 'Shade Opacity: 40%', 'bne' ),
		'0.45'	=>	esc_html__( 'Shade Opacity: 45%', 'bne' ),
		'0.5'	=>	esc_html__( 'Shade Opacity: 50%', 'bne' ),
		'0.55'	=>	esc_html__( 'Shade Opacity: 55%', 'bne' ),
		'0.6'	=>	esc_html__( 'Shade Opacity: 60%', 'bne' ),
		'0.65'	=>	esc_html__( 'Shade Opacity: 65%', 'bne' ),
		'0.7'	=>	esc_html__( 'Shade Opacity: 70%', 'bne' ),
		'0.75'	=>	esc_html__( 'Shade Opacity: 75%', 'bne' ),
		'0.8'	=>	esc_html__( 'Shade Opacity: 80%', 'bne' ),
		'0.85'	=>	esc_html__( 'Shade Opacity: 85%', 'bne' ),
		'0.9'	=>	esc_html__( 'Shade Opacity: 90%', 'bne' ),
		'0.95'	=>	esc_html__( 'Shade Opacity: 95%', 'bne' )
	);
	return apply_filters( 'bne_recognized_background_shade_opacity', $default );
}


/*
 * 	Get Theme Colors
 *
 *	Passes an array of theme specific colors for the
 *	shortcode generator options, specifically for buttons.
 *
 *	@since 		v1.3.6
 *	@updated	v1.5
 *
*/
function bne_recognized_theme_colors() {
	$default = array( 
		'default',
		'primary',
		'warning',
		'danger',
		'success',
		'info',
		'black',
		'blue',
		'brown',
		'dark_blue',
		'dark_brown',
		'dark_green',
		'green',
		'mauve',
		'orange',
		'pearl',
		'pink',
		'purple',
		'red',
		'slate_grey',
		'silver',
		'steel_blue',
		'teal',
		'yellow',
		'wheat'
	);
	return apply_filters( 'bne_recognized_theme_colors', $default );
}



/*
 * 	Get array of framework icons
 *
 *	returns an array of vector icons (Font Awesome) to be used within the
 *	BNE Framework and is cached.
 *
 * 	@since 		v1.5
 *	@updated	v1.6.1
 *
 */
function bne_get_icons( $type = 'vector' ) {
	$cache_name = 'bne_'.get_template().'_v'.BNE_FRAMEWORK_VERSION.'_'.$type.'_icons';
	$icons = get_transient( $cache_name );
	if( !$icons ) {
		$path = bne_enqueue_fontawesome( 'icons' );
		$fa_get_icons = wp_remote_get( esc_url_raw( $path ) );
		$fa_get_icons = json_decode( wp_remote_retrieve_body( $fa_get_icons ), true );
		
		$icons = array();
		$keywords = array();
		foreach( $fa_get_icons as $index => $icon ) {
			
			$keywords = array( $icon['label'] );
			if( isset( $icon['search']['terms'] ) ) {
				if( $icon['search']['terms'] ) {
					$keywords = array_merge( $keywords, $icon['search']['terms'] );
				}
			}
			
			$icons[$index] = array(
				'label' 	=> 	str_replace( '-', ' ', $icon['label'] ),
				'id' 		=> 	$index,
				'styles'	=>	$icon['svg'],
				'keywords'	=>	implode( ',', $keywords ),
			);
		
		}
		
		// Cache result
		if( !empty($icons) ) {
			set_transient( $cache_name, $icons, 86400 ); // 1 day
		}
	}
	
	return apply_filters( 'bne_'.$type.'_icons', $icons );
}



/*
 *	Add body classes
 *	Adds additional classes to the body tag of each page.
 *
 * 	@since 		v1.0
 *	@updated	v2.2.8
 *
*/
function bne_body_classes( $classes ) {

	global $post;

	// Get Options
	$options = bne_get_page_option();

	/*
	 *	Add Browser and device type classes to body tag
	 *	allowing individual CSS customizations if needed.
	*/
	// Get current user agent
	$browser = $_SERVER[ 'HTTP_USER_AGENT' ];

	// OS class
	if( preg_match( "/Mac/", $browser ) ) {
		$classes[] = 'mac';
	} elseif( preg_match( "/Windows/", $browser ) ) {
		$classes[] = 'windows';
	} elseif( preg_match( "/Linux/", $browser ) ) {
		$classes[] = 'linux';
	} else {
		$classes[] = 'unknown-os';
	}

	// Browser class
	if( preg_match( "/Edge/", $browser ) ) {
		$classes[] = 'ie12 msedge';
	} elseif( preg_match( "/Chrome/", $browser ) ) {
		$classes[] = 'chrome';
	} elseif( preg_match( "/Safari/", $browser ) ) {
		$classes[] = 'safari';
	} elseif( preg_match( "/Opera/", $browser ) ) {
		$classes[] = 'opera';
	} elseif( preg_match( "/MSIE/", $browser ) ) {

		// Internet Explorer... ugh, kill me now.
		$classes[] = 'msie';
		if( preg_match( "/MSIE 6.0/", $browser ) ) {
			$classes[] = 'ie6';
		} elseif( preg_match( "/MSIE 7.0/", $browser ) ) {
			$classes[] = 'ie7';
		} elseif( preg_match( "/MSIE 8.0/", $browser ) ) {
			$classes[] = 'ie8';
		} elseif( preg_match( "/MSIE 9.0/", $browser ) ) {
			$classes[] = 'ie9';
		} elseif( preg_match( "/MSIE 10.0/", $browser ) ) {
			$classes[] = 'ie10';
		} elseif( preg_match( "/MSIE 11.0/", $browser ) ) {
			$classes[] = 'ie11';
		}

	} elseif( preg_match( "/Firefox/", $browser ) && preg_match( "/Gecko/", $browser ) ) {
		$classes[] = 'firefox';
	} else {
		$classes[] = 'unknown-browser';
	}

	// Add group-blog to blogs with more than 1 published author.
	if( is_multi_author() ) {
		$classes[] = 'group-blog';
	}

	// Template and Sidebar Layout Type
	$classes[] = 'sidebar-layout-'.$options['sidebar_layout'];

	// Has post thumbnail class
	if( '' != get_the_post_thumbnail() ) {
		$classes[] = 'has-post-thumbnail';
	}

	// Header Status
	if( 'transparent' == $options['header_display'] )  {
		$classes[] = 'header-transparent';
		$classes[] = 'header-menu-'.get_theme_mod( 'menu_location', 'default' );
	} elseif( 'hide' == $options['header_display'] )  {
		$classes[] = 'header-off';
	}
	
	// Footer Status
	if( 'hide' == $options['footer_display'] )  {
		$classes[] = 'footer-off';
	}
	
	// Layout Type class ( boxed / wide )
	$classes[] = get_theme_mod( 'layout', 'wide_layout' );

	// Sticky Menu class
	if( 'true' == get_theme_mod( 'sticky_header', 'false' ) ) {
		$classes[] = 'has-sticky-menu';
	}

	// Mobile Nav Type class
	$responsive_menu = get_theme_mod( 'responsive_menu', 'graphical_responsive_menu' );
	if( 'graphical_responsive_menu' == $responsive_menu ) {
		$classes[] = 'bne-graphical-mobile-nav';

	} elseif( 'side_responsive_menu' == $responsive_menu ) {
		$classes[] = 'bne-side-mobile-nav';
	
	} elseif( 'disable_responsive_menu' == $responsive_menu ) {
		$classes[] = 'bne-disable-mobile-nav';
	}
	
	// Center Menu class
	if( true == get_theme_mod( 'menu_center' ) ) {
		$classes[] = 'header-menu-center';
	}

	// Dark Theme Styles (frontend)
	if( true == apply_filters( 'bne_dark_theme_mode', false ) ) {
	//if( true == get_theme_mod( 'dark_theme' ) ) {
		$classes[] = 'bne-dark-theme';
	}
	
	// Support for Max Mega Menu
	if( class_exists( 'Mega_Menu' ) ) {
		$classes[] = 'has-max-mega-menu';
	}

	return $classes;

}
add_filter( 'body_class', 'bne_body_classes' );



/*
 *	Add Column Classes
 *	Configures column classes based on layout and location
 *
 * 	@since 		v1.0
 *	@updated	v2.0.3
 *
*/
function bne_get_column_class( $column = '', $location = 'main' ) {

	/*
	 * 	Header Columns
	*/
	if( $location == 'header' ) {
		$layout = get_theme_mod( 'header_columns', 'half_half' );
		if( $layout == 'full_header' ) {
			$sizes = '12-12';
		} elseif( $layout == 'half_half' ) {
			$sizes = '6-6';
		} elseif( $layout == 'third_two_thirds' ) {
			$sizes = '4-8';
		} elseif( $layout == 'fourth_three_fourth' ) {
			$sizes = '3-9';
		}

		if( $sizes ) {
			$sizes = explode('-', $sizes ); 
			$logo_col = $sizes[0];
			$content_col = $sizes[1];
		}

		if( $column == 'left-header' ) {
			$classes = 'col-md-'.$logo_col;
		} else {
			$classes = 'col-md-'.$content_col;
		}
		
		// Overide for header sidebar layout
		$site_layout = get_theme_mod( 'layout', 'wide_layout' );
		if( $site_layout == 'header_sidebar_layout' ) {
			if( $column == 'left-header' || $column == 'right-header' ) {
				$classes = 'col-12';
			}
		}
		
	}
		
	/*
	 * 	Main / Sidebar Columns
	*/
	if( $location == 'main' ) {
		$layout = bne_get_page_option( 'sidebar_layout' );
		
		// Get Sidebar Layout Sizes and break point (Bootstrap)
		$content_col = '8';
		$sidebar_col = '4';
		$break = apply_filters( 'bne_sidebar_layout_responsive', 'md' ); // Bootstrap responsive break
		$sizes = apply_filters( 'bne_sidebar_layout_width', '8-4' ); // Must return #-#
	
		if( $sizes ) {
			$sizes = explode('-', $sizes ); 
			$content_col = $sizes[0];
			$sidebar_col = $sizes[1];
		}
	
		// Content Column Class
		if( $column == 'content' ) {
	
			// No sidebars
			if( $layout == 'no-sidebar' ) { 
				$classes = 'col-'.$break.'-12';
			} 
			
			// Double Sidebar Layout
			elseif( $layout == 'double-sidebar' ) { 
				$classes = 'col-'.$break.'-6';
			} 
			
			// Single Left Sidebar Layout
			elseif( $layout == 'left-sidebar' ) { 
				$classes = 'col-'.$break.'-'.$content_col;
			} 
			
			// Single Right Sidebar Layout
			else { 
				$classes = 'col-'.$break.'-'.$content_col;
			}
	
		}
	
		// Left Sidebar Column Class
		elseif( $column == 'sidebar-left' ) {
			$classes = ( $layout == 'double-sidebar' ) ? 'col-'.$break.'-3 order-'.$break.'-first' : 'col-'.$break.'-'.$sidebar_col.' order-'.$break.'-first';
		}
	
		// Right Sidebar Column
		elseif( $column == 'sidebar-right' ) {
			$classes = ( $layout == 'double-sidebar' ) ? 'col-'.$break.'-3' : 'col-'.$break.'-'.$sidebar_col;
		}		
	}
	
	return apply_filters( 'bne_get_column_classes', $classes, $column, $layout );
}



/*
 *	List pages as a main navigation menu when user has not
 *	set one under Appearance > Menus in the WordPress admin panel.
 *
 * 	@since v1.0
 *
*/
if( !function_exists( 'bne_primary_menu_fallback' ) ) :
	function bne_primary_menu_fallback() {
		echo '<ul id="primary-menu" class="sf-menu">';
		wp_list_pages('title_li=');
		echo '</ul>';
	}
endif;



/*
 *	Get our wp_nav_menu() fallback, wp_page_menu(), to show a home link.
 *
 * 	@since v1.0
 *
*/
function bne_page_menu_args( $args ) {
	$args['show_home'] = true;
	return $args;
}
add_filter( 'wp_page_menu_args', 'bne_page_menu_args' );



/*
 * 	Filter framework items into wp_nav_menu() output.
 *	Allows for Font Awesome icons, bootstrap headings and dividers.
 *
 * 	@since 		v1.0
 *	@updated 	v2.2.1
 *
*/
function bne_nav_menu_start_el( $item_output, $item, $depth, $args ) {

	// Arrow indicators on drop down menus
	if( in_array( 'menu-item-has-children', $item->classes ) && strpos( $args->menu_class, 'sf-menu' ) !== false ) {

		$direction = ( $depth > 0 ) ? 'right' : 'down';
		$indicator = sprintf( '<i class="sf-sub-indicator bne-fa fas fa-caret-%s" aria-hidden="true"></i>', $direction );
		$item_output = str_replace( '</a>', $indicator.'</a>', $item_output );
	}

	// Allow bootstrap "nav-header" class in menu items.
	// Note: For primary navigation will only work on levels 2-3
	// (1) ".sf-menu li li.nav-header" 	=> Primary nav dropdowns
	// (2) ".menu li.nav-header" 		=> Standard custom menu widget
	if( in_array( 'nav-header', $item->classes )  ) {

		$header = sprintf( '<span>%s</span>', apply_filters( 'the_title', $item->title, $item->ID ) );

		if( strpos( $args->menu_class, 'sf-menu' ) !== false ) {
			// Primary Navigation
			if( $depth > 0 ) {
				$item_output = $header;
			}
		} else {
			$item_output = $header;
		}
	}

	// Allow bootstrap "divider" class in menu items.
	// Note: For primary navigation will only work on levels 2-3
	if( in_array( 'divider', $item->classes )  ) {

		if( strpos( $args->menu_class, 'sf-menu' ) !== false ) {
			// Primary Navigation
			if( $depth > 0 ) {
				$item_output = '';
			}
		} else {
			$item_output = '';
		}
	}

	// Font Awesome icons in menu items
	$icons = array();
	foreach( $item->classes as $class ) {
		if( strpos( $class, 'menu-icon-' ) !== false ) {
			$icons[] = str_replace( 'menu-icon-', '', $class );
		}
	}

	if( $icons ) {
		foreach( $icons as $icon ) {
			$text = apply_filters( 'the_title', $item->title, $item->ID );
			$icon_output = sprintf( '<i class="bne-fa fa fa-%s menu-icon" aria-hidden="true"></i>', $icon );
			$icon_output = apply_filters( 'bne_menu_icon', $icon_output, $icon );
			$item_output = str_replace( $text, $icon_output.$text, $item_output );
		}
	}
	
	// Title & Description
	if( true == apply_filters( 'bne_nav_menu_item_descriptions', false ) ) {
		if( $item->title ) {
			$text = '<span class="menu-label">'.$item->title.'</span>';
			
			if( $item->description ) { 
				$text = $text.'<span class="menu-description">'.$item->description.'</span>';
			}
			
			$item_output = str_replace( $item->title, $text, $item_output );
		}
	}
	
	return $item_output;
}
add_filter( 'walker_nav_menu_start_el', 'bne_nav_menu_start_el', 10, 4 );



/*
 * 	Menu Classes
 *	Adds a unique menu class to signify which is
 *	active and parent.
 *
 * 	@since 		v1.4.2
 *
*/
function bne_nav_menu_css_class( $classes, $item ) {
     
     if( in_array( 'current-menu-item', $classes ) || in_array( 'current-menu-parent', $classes ) ){
        $classes[] = 'active ';
     }
     
     return $classes;
}
add_filter( 'nav_menu_css_class' , 'bne_nav_menu_css_class' , 10 , 2 );



/*
 *	Breadcrumb Wrapper
 *
 * 	@since 		v1.0
 *	@updated 	v1.6
 *
*/
if( !function_exists( 'bne_breadcrumbs_default' ) ) :
	function bne_breadcrumbs_default() {

		// Get Options
		$breadcrumbs = bne_get_page_option( 'breadcrumbs' );

		// Output
		if( 'hide' != $breadcrumbs ) {

			echo '<div class="breadcrumbs">';
				echo '<div class="breadcrumbs-inner content-area-width">';
					echo '<div class="breadcrumbs-content">';
						echo '<ul class="breadcrumb">';

							// Check for Yoast Breadcrumbs
							$yoast_seo = get_option( 'wpseo_internallinks' );
							if( function_exists( 'yoast_breadcrumb') && $yoast_seo['breadcrumbs-enable'] === true ) {
								yoast_breadcrumb();
							// Theme Breadcrumbs
							} else {
								bne_breadcrumb_trail();
							}

						echo '</ul>';
					echo '</div><!-- .breadcrumbs-content (end) -->';
				echo '</div><!-- .breadcrumbs-inner (end) -->';
			echo '</div><!-- .breadcrumbs (end) -->';
		}

	}
endif;



/*
 *	Breadcrumb Trail
 *
 * 	@since 		v1.0
 *	@updated	v2.0
 *
*/
function bne_breadcrumb_trail() {
	global $post, $post_type;

	if( !is_home() ) {

		// Breadcrumb Trail Before Filter
		echo apply_filters( 'bne_breadcrumb_trail_before', false );

		// Home Link
		echo '<li class="breadcrumb-item"><a href="'.home_url().'" title="'.get_bloginfo('name').'" aria-label="Home"><i class="bne-fa fas fa-home" aria-hidden="true"></i></a></li>';

		// Place Category Terms if on an archive or single post
		if( is_category() || ( has_category() && is_single() ) ) {
			echo '<li class="breadcrumb-item">';
			the_category(', ');
			echo '</li>';
		}

		// Breadcrumb Trail Addon Filter
		echo apply_filters( 'bne_breadcrumb_trail_addon', false );

		// Archives
		if( is_archive() && !is_category() ) {

			// Day
			if( is_day() ) {
				echo '<li class="breadcrumb-item active">'.esc_html__( 'Archives for', 'bne' ).' '.get_the_time('F jS, Y').'</li>';
			}

			// Month
			elseif( is_month() ) {
				echo '<li class="breadcrumb-item active">'.esc_html__( 'Archives for', 'bne' ).' '.get_the_time('F, Y').'</li>';
			}

			// Year
			elseif( is_year() ) {
				echo '<li class="breadcrumb-item active">'.esc_html__( 'Archives for', 'bne' ).' '.get_the_time('Y').'</li>';
			}

			// Author
			elseif( is_author() ) {
				echo '<li class="breadcrumb-item active">'.esc_html__( 'Articles posted by', 'bne' ).' '.get_the_author().'</li>';
			}

			// Tags
			elseif( is_tag() ) {
				echo '<li class="breadcrumb-item active">'.esc_html__( 'Post tagged with', 'bne' ).' "';
					echo single_tag_title();
				echo '"</li>';
			}

			// Custom Taxonomies
			elseif( is_tax() ) {
				$term = get_term_by( 'slug', get_query_var( 'term' ), get_query_var( 'taxonomy' ) );
				echo '<li class="breadcrumb-item">' . $term->name . '</li>';
			}


			// Add Support for Custom Post Types
			// WooCommerce is handled with the bne_breadcrumb_trail_addon filter
			elseif( get_post_type() != 'post' && get_post_type() != 'product' ) {

				$post_type_object = get_post_type_object( $post_type );
				$archive_link = get_post_type_archive_link( $post_type );
				if( $post_type_object ) {
					echo '<li class="breadcrumb-item active">'.$post_type_object->labels->name.'</li>';
				}
			}

		}

		// Pages
		elseif( is_page() ) {
			if( $post->post_parent ) {
				$ancestors = get_post_ancestors( $post->ID );
				$title = get_the_title();
				foreach( array_reverse( $ancestors ) as $ancestor ) {
					echo '<li class="breadcrumb-item"><a href="'.get_permalink($ancestor).'" title="'.get_the_title( $ancestor ).'">'.get_the_title( $ancestor ).'</a></li>';
				}
				echo '<li class="breadcrumb-item active">'.$title.'</li>';
			} else {
				echo '<li class="breadcrumb-item active">'.get_the_title().'</li>';
			}
		}

		// Single
		elseif( is_single() ) {

			// Add Support for Custom Post Types
			// WooCommerce is handled with the bne_breadcrumb_trail_addon filter
			if( get_post_type() != 'post' && get_post_type() != 'product' ) {

				$post_type_object = get_post_type_object( $post_type );
				$archive_link = get_post_type_archive_link( $post_type );

				if( $post_type_object ) {
					echo '<li class="breadcrumb-item"><a href="'.$archive_link.'" title="'.$post_type_object->labels->name.'">'.$post_type_object->labels->name.'</a></li>';
				}

			}

			echo '<li class="breadcrumb-item active">'.get_the_title().'</li>';

		}

		// Search
		elseif( is_search() ) {
			echo '<li class="breadcrumb-item active">'.esc_html__( 'Search Results', 'bne' ).'</li>';
		}

		// 404
		elseif( is_404() ) {
			echo '<li class="breadcrumb-item active">'.esc_html__( 'Not Found', 'bne' ).'</li>';
		}

		// Finally, add page number if is paged
		if( get_query_var('paged') ) {
			echo ' ('.esc_html__( 'page','bne' ).' '.get_query_var('paged').')';
		}

		// After Trail Filter
		echo apply_filters( 'bne_breadcrumb_trail_after', false );

	} // End !is_home()

}




/*
 *	Page/Post Title Wrapper
 *
 *	Outputs the default page/post heading that is found
 *	in content-page.php and content-single.php templates.
 *
 *	@since v2.2.9
 *
*/
function bne_page_heading_default() {
	$classes = 'entry-header page-heading';
	$subtitle = false;
	$title = false;
	
	// Pages & Post
	if( is_singular() ) {
		$display = get_post_meta( get_the_id(), 'bne_title', true );
		
		if( $display == 'show_center' ) {
			$classes .= ' text-center';
		} elseif( $display == 'show_right' ) {
			$classes .= ' text-right';
		}

		if( 'hide' != $display ) {
			$title = get_the_title();
		}
		
	// Blog Posts (Settings > Reading)
	} elseif( is_home() ) {
		$posts_page = get_option( 'page_for_posts' );
		$display = get_post_meta( $posts_page, 'bne_title', true );
			
		if( $posts_page && 'hide' != $display ) {
			$title = get_post( $posts_page )->post_title;
		}
		
		if( $display == 'show_center' ) {
			$classes .= ' text-center';
		} elseif( $display == 'show_right' ) {
			$classes .= ' text-right';
		}


	// Archives
	} elseif( is_archive() ) {
		$show_title = get_theme_mod( 'post_archive_title', true );
		$show_description = get_theme_mod( 'post_archive_description', true );
		
		if( $show_title == true ) {
			$title = single_term_title( '', false );
		}
		
		if( $show_description == true ) {
			$subtitle = term_description();
		}
	
	// Search
	} elseif( is_search() ) {
		$title = __( 'Search Results for: ', 'bne' ).get_search_query();
	}
	

	// Allow developers to add a custom heading 
	$title = apply_filters( 'bne_page_heading_text', $title );
	$subtitle = apply_filters( 'bne_page_heading_subtext', $subtitle );
	
	// Echo Title
	if( $title ) {
		echo '<header class="'.$classes.'">';
			echo '<h1 class="entry-title">'.$title.'</h1>';
			
			if( $subtitle) {
				echo '<div class="term-description">'.$subtitle.'</div>';
			}
			
		echo '</header><!-- .entry-header -->';
		
	}
	
}




/*
 *	Archive Content Before
 *
 *	Includes the Blog page title for the Post Page template
 *	(Settings > Reading). Includes the archive taxonomy title 
 *	and description if entered. Devs can extend with custom content
 *	before the archive loop.
 *
 *	@since 	v2.1.4
 *	@updatd	v2.2.5
 *
 *	@depreciated v2.2.7
 *
*/
function bne_archive_content_before() {
	$show_title = get_theme_mod( 'post_archive_title', true );
	$show_description = get_theme_mod( 'post_archive_description', true );
	$output = '';
	$title = '';
	$description = '';
	
	if( is_home() ) {
		$posts_page = get_option( 'page_for_posts' );
		if( $posts_page ) {
			$title = get_post( $posts_page )->post_title;
		}
	} elseif( is_archive() ) {
		$title = single_term_title( '', false );
		$description = term_description();
	} 
	
	if( $show_title == true && $title ) {
		$output .= sprintf( '<header class="entry-header page-heading term-heading mb-3"><h1 class="entry-title">%s</h1></header>', $title );
	}
	
	if( $show_description == true && $description ) {
		$output .= sprintf( '<div class="term-description mb-5">%s</div>', $description );
	}
	
	echo apply_filters( 'bne_archive_content_before', $output );
	
}




/*
 *	Post Excerpt Length
 *	Sets the post excerpt length to 25 characters.
 *
 * 	@since 		v1.0
 *	@updated 	v1.5.2
 *
*/
function bne_excerpt_length( $length ) {
	return apply_filters( 'bne_excerpt_length', '25' );
}
add_filter( 'excerpt_length', 'bne_excerpt_length' );



/*
 * 	WP Page Link Args
 * 	When using wp_link_pages(), match surrounding markup
 * 	to bne_pagination() and integration of Bootstrap.
 *
 * 	@since 		v1.0
 *	@updated	v2.0
 *
*/
function bne_link_pages_args( $args ) {

	global $wp_version;

	// Add Bootstrap surrounding markup
	$args['before'] = '<div class="pagination">';
	$args['after'] = "</div>\n";

	return $args;
}
add_filter( 'wp_link_pages_args', 'bne_link_pages_args' );



/*
 * 	WP Page Link
 *
 * 	When using wp_link_pages(), match individual button markup
 * 	to bne_pagination() and integration of Bootstrap.
 *
 * 	@since 		v1.0
 *	@updated	v2.0
 *
*/
function bne_link_pages_link( $link, $i ) {

	global $page;

	if( $page == $i ) {
		$link = sprintf( '<span class="page-item active"><a href="%s" class="page-link">%s</a>', get_pagenum_link($i), $i );
		$link = str_replace( '</a>', '</span></a>', $link );
	} else {
		
		$link = str_replace( '<a', '<span class="page-item"><a class="page-link"', $link );
		$link = str_replace( '</a>', '</span></a>', $link );
	}

	return $link;
}
add_filter( 'wp_link_pages_link', 'bne_link_pages_link', 10, 2 );



/*
 * 	Pagination
 *	found in single.php, index.php, archive.php, search.php
 *
 *	@param		$nav_id				Sets the css ID of the nav to be used for styling.
 *
 *	@param		$max_num_pages		pulls in the max_num_pages from the template query.
 *									If not specified, will default to $wp_query.
 *
 *	@since 		v1.0
 *	@updated	v2.0
 *
*/
if( !function_exists( 'bne_pagination' ) ) :
	function bne_pagination( $nav_id, $max_num_pages = '' ) {
		global $wp_query, $post;

		$big = 999999;
		
		// If no max is provided, use the $wp_query
		if( !$max_num_pages ) { $max_num_pages = $wp_query->max_num_pages; }
		
		// Don't print empty markup on single pages if there's nowhere to navigate.
		if( is_single() ) {
			$previous = ( is_attachment() ) ? get_post( $post->post_parent ) : get_adjacent_post( false, '', true );
			$next = get_adjacent_post( false, '', false );

			if( !$next && !$previous )
				return;
		}

		// Don't print empty markup in archives if there's only one page.
		if( $max_num_pages < 2 && ( is_home() || is_archive() || is_search() ) )
			return;

		?>

			<nav role="navigation" id="<?php echo esc_attr( $nav_id ); ?>" class="bne-pagination">

				<?php if( is_single() ) { ?>
					<ul id="bne-single-post-nav" class="pager clearfix">
						<?php previous_post_link( '<li class="nav-previous previous">%link</li>', '<span class="meta-nav">' . _x( '&larr;', 'Previous post link', 'bne' ) . '</span> %title' ); ?>
						<?php next_post_link( '<li class="nav-next next">%link</li>', '%title <span class="meta-nav">' . _x( '&rarr;', 'Next post link', 'bne' ) . '</span>' ); ?>

					</ul>

				<?php } elseif( $max_num_pages > 1 ) { ?>

					<?php
						$links = paginate_links( array(
							'base'		=>	str_replace( $big, '%#%', esc_url( get_pagenum_link( $big ) ) ),
							'format'	=> 	'?paged=%#%',
							'current' 	=> 	max( 1, get_query_var('paged') ),
							'type'		=> 	'array',
							'prev_next' => 	true,
							'prev_text'	=> 	'<i class="bne-fa fas fa-angle-double-left" aria-label="Previous"></i>',
							'next_text'	=> 	'<i class="bne-fa fas fa-angle-double-right" aria-label="Next"></i>',
							'total' 	=> 	$max_num_pages
						) );

						if( is_array( $links ) ) {
							$paged = ( get_query_var('paged') == 0 ) ? 1 : get_query_var('paged');
							echo '<ul class="pagination">';
							foreach( $links as $link ) {
								echo '<li class="page-item '.(strpos($link, 'current') !== false ? 'active' : '').'"> ' . str_replace( 'page-numbers', 'page-link', $link ) . '</li>';
        					}
							echo '</ul>';
        				}


					?>

				<?php } ?>

			</nav><!-- .pagination (end) -->

		<?php
	}
endif; // bne_pagination



/*
 * 	Template for comments and pingbacks.
 *	Used as a callback by wp_list_comments() for displaying the comments.
 *
 *	@since 		v1.0
 *	@updated	v1.5.14
 *
*/
if( !function_exists( 'bne_comment' ) ) :
	function bne_comment( $comment, $args, $depth ) {
		$GLOBALS['comment'] = $comment;

		if( 'pingback' == $comment->comment_type || 'trackback' == $comment->comment_type ) : ?>

		<li id="comment-<?php comment_ID(); ?>" <?php comment_class( 'media' ); ?>>
			<div class="comment-body">
				<?php _e( 'Pingback:', 'bne' ); ?> <?php comment_author_link(); ?> <?php edit_comment_link( esc_html__( 'Edit', 'bne' ), '<span class="edit-link">', '</span>' ); ?>
			</div>

		<?php else : ?>

		<li id="comment-<?php comment_ID(); ?>" <?php comment_class( empty( $args['has_children'] ) ? '' : 'parent' ); ?>>
			<div id="div-comment-<?php comment_ID(); ?>" class="comment-body">

				<!-- Meta -->
				<div class="comment-meta">

					<!-- Author -->
					<div class="comment-author vcard">
						<div class="pull-left comment-avatar">
							<?php if( 0 != $args['avatar_size'] ) echo get_avatar( $comment, $args['avatar_size'] ); ?>
						</div>
						<?php printf( __( '%s <span class="says">says:</span>', 'bne' ), sprintf( '<b class="fn">%s</b>', get_comment_author_link() ) ); ?>
					</div>

					<!-- Metadata -->
					<div class="comment-metadata">
						<a href="<?php echo esc_url( get_comment_link( $comment->comment_ID ) ); ?>">
							<time datetime="<?php comment_time( 'c' ); ?>">
								<?php printf( _x( '%1$s at %2$s', '1: date, 2: time', 'bne' ), get_comment_date(), get_comment_time() ); ?>
							</time>
						</a>

						<!-- Edit Link -->
						<?php edit_comment_link( 
							sprintf( esc_html__('%s Edit', 'bne'), '<span style="margin-left: 5px;" class="glyphicon glyphicon-edit"></span>' ),
							'<span class="edit-link">',
							'</span>'
						); ?>
							
						<!-- Reply Link -->
						<?php comment_reply_link(
							array_merge(
								$args, array(
									'before'		=>	'<span class="comment-reply reply">',
									'after'			=>	'</span>',
									'reply_text'	=>	sprintf( esc_html__('%s Reply', 'bne'), '<i class="bne-fa fas fa-reply"></i> ' ),
									'add_below' 	=> 	'div-comment',
									'depth' 		=> 	$depth,
									'max_depth' 	=> 	$args['max_depth']
								)
							)
						); ?>
					
					</div><!-- .comment-metadata (end) -->
				</div><!-- .comment-meta (end) -->

				<?php if( '0' == $comment->comment_approved ) : ?>
					<p class="comment-awaiting-moderation"><?php _e( 'Your comment is awaiting moderation.', 'bne' ); ?></p>
				<?php endif; ?>

				<div class="comment-content">
					<?php comment_text(); ?>
				</div><!-- .comment-content -->

			</div><!-- .comment-body (end) -->

		<?php
		endif;
	}
endif; // ends check for bne_comment()



/*
 * 	Comment Form Default Fields Layout
 *	Customizes the default comment form fields and integrates
 *	bootstrap css.
 *
 *	@since 		v1.3.4
 *	@updated	v2.0
 *
*/
function bne_comment_form_layout( $fields ) {
    
    // WP Requirements
	$commenter = wp_get_current_commenter();
	$req       = get_option( 'require_name_email' );
	$aria_req  = ( $req ? " aria-required='true'" : '' );
	$html_req  = ( $req ? " required='required'" : '' );
	$html5     = 'html5';
	$consent  = empty( $commenter['comment_author_email'] ) ? '' : ' checked="checked"';
	
	$name_label  	= esc_html__( 'Name', 'bne' );
	$email_label	= esc_html__( 'Email', 'bne' );
	$website_label  = esc_html__( 'Website', 'bne' );
	$cookie_label  = esc_html__( 'Save my name, email, and website in this browser for the next time I comment.', 'bne' );
	
	//bne_printr($fields);
	$fields = array();
	
	// Open Row
	$fields[] = '<div class="row">';
	
	// Author
	$fields['author'] = '
		<div class="col-sm-4">
			<div class="comments_field comment-form-author form-group">
				<label for="author" class="sr-only">'.$name_label.( $req ? ' <span class="required">*</span>' : '' ).'</label>
				<input id="author" name="author" class="form-control" type="text" value="' . esc_attr( $commenter['comment_author'] ) . '" size="30" '.$aria_req.$html_req.' placeholder="'.$name_label.' *"/>
			</div>
		</div><!-- .col (end) -->
	';
	
	// Email
	$fields['email'] = '
		<div class="col-sm-4">
			<div class="comments_field comment-form-email form-group">
				<label for="email" class="sr-only">'.$email_label.( $req ? ' <span class="required">*</span>' : '' ).'</label>
				<input id="email" name="email" class="form-control" '.( $html5 ? 'type="email"' : 'type="text"' ).' value="'.esc_attr(  $commenter['comment_author_email'] ).'" size="30" aria-describedby="email-notes"'.$aria_req.$html_req.' placeholder="'.$email_label.' *"/>
			</div>
		</div><!-- .col (end) -->
	';

	// URL
	$fields['url'] = '
		<div class="col-sm-4">
			<div class="comments_field comment-form-url form-group">
				<label for="url" class="sr-only">'.$website_label.'</label>
				<input id="url" name="url" class="form-control" '.( $html5 ? 'type="url"' : 'type="text"' ).' value="'.esc_attr( $commenter['comment_author_url'] ).'" size="30" placeholder="'.$website_label.'"/>
			</div>
		</div><!-- .col (end) -->
	';
	
	// Close row
	$fields[] = '</div><!-- .row (end) -->';


	// Cookies
	$fields['cookies'] = '<div class="comments_field comment-form-cookies-consent form-group">
		<label for="wp-comment-cookies-consent">
			<input id="wp-comment-cookies-consent" name="wp-comment-cookies-consent" type="checkbox" value="yes" '.$consent.'>
			'.$cookie_label.'
		</label>
	</div>';

	return $fields;  
	   
}
add_filter( 'comment_form_default_fields', 'bne_comment_form_layout' );



/*
 * 	Blog post meta for date, author, category, and number of comments.
 *	found in content-single.php
 *
 *	@since 		v1.0
 *	@udpated 	v2.1
 *
*/
if( !function_exists( 'bne_post_meta' ) ) :
	function bne_post_meta( $sep = '', $return = array( 'date', 'author', 'category', 'comments' ), $location = 'header' ) {

		// Opening Container
		$output = '<div class="entry-meta meta-'.$location.'">';

			// Separator
			if( !$sep ) {
				$sep = apply_filters( 'bne_meta_separator', '<span class="sep"> / </span>' );
			}
	
			// Date
			$date = sprintf('<span class="entry-date"><time class="updated" datetime="%s"><i class="bne-fa fas fa-clock" aria-hidden="true"></i> %s</time></span>', get_the_time('c'), get_the_time( get_option( 'date_format' ) ) );
		
			// Author
			$author_url = get_author_posts_url( get_the_author_meta('ID') );
			$author_title = sprintf( esc_html__( 'View all posts by %s', 'bne' ), get_the_author() );
			$author = sprintf( '<span class="author vcard"><i class="bne-fa fas fa-user" aria-hidden="true"></i> <a class="url fn n" href="%s" title="%s" rel="author">%s</a></span>', $author_url, $author_title, get_the_author() );

			// Category
			$category = null;
			if( has_category() ) {
				$category .= sprintf( '<span class="category"><i class="bne-fa fas fa-folder" aria-hidden="true"></i> %s</span>', get_the_category_list(', ') );
			}

			// Comments
			$comments = null;
			if( comments_open() ) {
				$comments .= '<span class="comments-link">';
					ob_start();
					comments_popup_link( '<span class="leave-reply">no comments</span>', '1 comment', '% comments' );
					$comment_link = ob_get_clean();
					$comments .= sprintf( '<i class="bne-fa fas fa-comment" aria-hidden="true"></i> %s', $comment_link );
				$comments .= '</span>';
			}

			// Allow hijacking the meta order of the $return
			$return = apply_filters( 'bne_meta_order', $return, $location );
			
			// Setup meta content based on return
			$meta = null;
			if( $return ) {
				
				// Return as a sting
				if( is_string( $return ) ) {
					
					// If single known item, use that and go to final output.
					if( isset( $$return ) ) {
						$meta = $$return;
					
					// Backwards Compatibility. < v1.5.4 used a string with underscores "_".
					// Lets use that by turning it into an array for the new output below.
					} else {
						$return = explode( '_', $return );
					}
				}

				// Return as an array to allow any custom order. Let's loop through it
				// and assign the correct meta markup.
				if( is_array( $return ) ) {
					foreach( $return as $meta_order ) {
						if( 'date' == $meta_order ) { $meta_content = $date;
						} elseif( 'author' == $meta_order ) { $meta_content = $author;
						} elseif( 'category' == $meta_order && $category ) { $meta_content = $category;
						} elseif( 'comments' == $meta_order && $comments ) { $meta_content = $comments;
						} else { continue; }
						$meta[] = $meta_content;
					}
					if( is_array( $meta ) ) {
						$meta = implode( $sep, $meta );
					}
				
				}
	
			}
			
			// Output complete meta
			$output .= apply_filters( 'bne_meta_output', $meta, $sep, $date, $author, $category, $comments, $location );

		$output .= '</div><!-- .entry-meta -->';

		return $output;

	}
endif;



/*
 * 	Outputs main blog post tags
 *	found in content-single.php
 *
 *	@since 		v1.0
 *	@updated	v1.5.14
 *
*/
if( !function_exists( 'bne_blog_tags' ) ) :
	function bne_blog_tags() {
		the_tags( '<span class="tags"><i class="bne-fa fas fa-tags"></i> ', ', ', '</span>' );
	}
endif;



/*
 * 	Sub Meta Wrapper
 *
 *	Displays social share buttons and post meta in the footer
 *	of the blog post.
 *
 *	@since 		v1.3.2
 *	@updated	v2.1
 *
*/
function bne_sub_meta_wrapper() {
	?>
	<div class="sub-meta-wrapper clearfix">
		
		<?php 
			$share = get_theme_mod( 'post_share_buttons', 'show' );
			if( 'show' == $share ) { 
				bne_share_buttons(); 
			}
		?>
		
		<div class="info">
			<?php //echo bne_post_meta( null, $return = array( 'category', 'comments' ), $location = "footer" ); ?>
			<?php the_tags( '<div class="bne-tags tags">', ' ', '</div>' ); ?>
		</div>
	
	</div>
	<?php
}



/*
 * 	Post Share Buttons
 *
 *	Adds Facebook, Twitter, Google+, and Email share buttons. Links
 *	will open in a new window if they include the class "popup".
 *	The configuration of the popup window is via bne-scripts.js
 *
 *	@url	http://petragregorova.com/articles/social-share-buttons-with-custom-icons/
 *
 *	@since 		v1.3.2
 *	@updated	v1.5.14
 *
*/
function bne_share_buttons() {
	
	// Get current page URL 
	$page_url = get_permalink();

	// Get current page title
	$page_title = str_replace( ' ', '%20', get_the_title() );
	
	// Thumbnail
	$thumbnail = wp_get_attachment_image_src( get_post_thumbnail_id( get_the_id() ), 'full' );

	// Construct sharing URL without using any script
	$share_urls = apply_filters('bne_get_share_button_links', array(
		'facebook'	=>	'https://www.facebook.com/sharer/sharer.php?u='.$page_url.'&title='.$page_title,
		'linkedin'	=>	'https://www.linkedin.com/shareArticle?mini=true&url='.$page_url.'&title='.$page_title,
		'twitter'	=>	'https://twitter.com/intent/tweet?status='.$page_title.'+'.$page_url,
		'email'		=>	'mailto:?subject='.$page_title.'&body='.$page_url,
		//'tumblr'	=>	'http://www.tumblr.com/share?v=3&u='.$page_url.'&t='.$page_title,
		//'pinterest'	=>	'http://pinterest.com/pin/create/bookmarklet/?media='.$thumbnail[0].'&url='.$page_url.'&is_video=false&description='.$page_title,
	), $page_url, $page_title, $thumbnail );

	// Add sharing button at the end of post content	
	$output = '<div class="share">';
		
		$output .= '<ul class="bne-share clearfix">';
			
			// Loop through Share links.
			foreach( $share_urls as $key => $url ) {
				
				// Defaults
				$fa_icon = $key;
				$fa_style = 'fab';
				$popup = 'popup';
				
				// Overrides
				if( 'email' == $key ) {
					$popup = null;
					$fa_style = 'fas';
					$fa_icon = 'envelope';
				} elseif( 'facebook' == $key ) {
					$fa_icon = 'facebook-f';
				}
				
				// Output link
				$output .= '<li><a href="'.$url.'" class="btn-share '.$key.' '.$popup.'" data-toggle="tooltip" data-placement="top" title="Share this via '.ucfirst( $key ).'" aria-label="Share this via '.ucfirst( $key ).'"><i class="bne-fa '.$fa_style.' fa-fw fa-'.$fa_icon.'"></i></a></li>';
			}

		$output .= '</ul>';
	$output .= '</div>';
	
	echo $output;

}



/*
 * 	Post Author Box
 *
 *	@since 		v1.4.6
 *	@updated	v2.1
 *
*/
if( !function_exists( 'bne_post_author_box' ) ) :
	function bne_post_author_box( $title = '' ) {
		
		// Author Details
		$author_id = get_the_author_meta( 'ID' );
		$author_name = get_the_author_meta( 'display_name' );
		$author_bio = wpautop( get_the_author_meta( 'description' ) );
		$author_link = get_author_posts_url( $author_id );

		$output = '<div class="post-author-box">';
			$output .= '<div class="author-inner-content clearfix">';
				
				// Avatar
				$output .= '<div class="author-avatar">';
					$output .= get_avatar( $author_id, 150, null, $author_name );
				$output .= '</div>';

				// Name
				$output .= '<h3 class="author-name">'.$author_name.'</h3>';

				// Bio
				$output .= '<div class="author-bio">'.$author_bio.'</div>';

				// Archive Link
				$output .= '<a href="'.$author_link.'" class="btn btn-default author-archive-link">'.esc_html__( 'View all post by', 'bne' ).' '.$author_name.'</a>';

			$output .= '</div><!-- .author-inner-content (end) -->';
		$output .= '</div><!-- .post-author-box (end) -->';
		
		echo $output;
	
	}
endif;



/*
 * 	Define Available Social Network Options to
 *	be used within Theme Options and the [contact_bar]
 *	shortcode.
 *
 *	@since 		v1.3.1
 *	@updated 	v2.0
 *
*/
function bne_get_social_media_sources( $request = null, $id = null ) {
	
	$sources = array(
		'amazon'		=> 	array(
			'id'			=>	'amazon',
			'label'			=>	'Amazon',
			'image'			=>	null,
			'fa_icon'		=>	'amazon',
			'fa_style'		=>	'fab',
			'fa_color'		=>	'#ff9900',
		),
		'delicious'		=> 	array(
			'id'			=>	'delicious',
			'label'			=>	'Delicious',
			'image'			=>	null,
			'fa_icon'		=>	'delicious',
			'fa_style'		=>	'fab',
			'fa_color'		=>	'#3399ff',
		),
		'deviantart'		=> 	array(
			'id'			=>	'deviantart',
			'label'			=>	'Deviant Art',
			'image'			=>	null,
			'fa_icon'		=>	'deviantart',
			'fa_style'		=>	'fab',
			'fa_color'		=>	'#05cc47',
		),
		'dribbble'		=> 	array(
			'id'			=>	'dribbble',
			'label'			=>	'Dribbble',
			'image'			=>	null,
			'fa_icon'		=>	'dribbble',
			'fa_style'		=>	'fab',
			'fa_color'		=>	'#ea4c89',
		),
		'email'			=> 	array(
			'id'			=>	'email',
			'label'			=>	'Email',
			'image'			=>	null,
			'fa_icon'		=>	'envelope',
			'fa_style'		=>	'fas',
			'fa_color'		=>	'#444444',
		),
		'facebook'		=> 	array(
			'id'			=>	'facebook',
			'label'			=>	'Facebook',
			'image'			=>	null,
			'fa_icon'		=>	'facebook-f',
			'fa_style'		=>	'fab',
			'fa_color'		=>	'#3b5998',
		),
		'flickr'		=> 	array(
			'id'			=>	'flickr',
			'label'			=>	'Flickr',
			'image'			=>	null,
			'fa_icon'		=>	'flickr',
			'fa_style'		=>	'fab',
			'fa_color'		=>	'#ff0084',
		),
		'foursquare'		=> 	array(
			'id'			=>	'foursquare',
			'label'			=>	'Foursquare',
			'image'			=>	null,
			'fa_icon'		=>	'foursquare',
			'fa_style'		=>	'fab',
			'fa_color'		=>	'#f94877',
		),
		'github'		=> 	array(
			'id'			=>	'github',
			'label'			=>	'Github',
			'image'			=>	null,
			'fa_icon'		=>	'github',
			'fa_style'		=>	'fab',
			'fa_color'		=>	'#4183c4',
		),
		'google'		=> 	array(
			'id'			=>	'google',
			'label'			=>	'Google+',
			'image'			=>	null,
			'fa_icon'		=>	'google',
			'fa_style'		=>	'fab',
			'fa_color'		=>	'#dc4e41',
		),
		'instagram'		=> 	array(
			'id'			=>	'instagram',
			'label'			=>	'Instagram',
			'image'			=>	null,
			'fa_icon'		=>	'instagram',
			'fa_style'		=>	'fab',
			'fa_color'		=>	'#F00075',
		),
		'linkedin'		=> 	array(
			'id'			=>	'linkedin',
			'label'			=>	'LinkedIn',
			'image'			=>	null,
			'fa_icon'		=>	'linkedin',
			'fa_style'		=>	'fab',
			'fa_color'		=>	'#0077b5',
		),
		'paypal'		=> 	array(
			'id'			=>	'paypal',
			'label'			=>	'PayPal',
			'image'			=>	null,
			'fa_icon'		=>	'paypal',
			'fa_style'		=>	'fab',
			'fa_color'		=>	'#003087',
		),
		'pinterest'		=> 	array(
			'id'			=>	'pinterest',
			'label'			=>	'Pinterest',
			'image'			=>	null,
			'fa_icon'		=>	'pinterest',
			'fa_style'		=>	'fab',
			'fa_color'		=>	'#bd081c',
		),
		'reddit'		=> 	array(
			'id'			=>	'reddit',
			'label'			=>	'Reddit',
			'image'			=>	null,
			'fa_icon'		=>	'reddit',
			'fa_style'		=>	'fab',
			'fa_color'		=>	'#ff4500',
		),
		'rss'		=> 	array(
			'id'			=>	'rss',
			'label'			=>	'RSS',
			'image'			=>	null,
			'fa_icon'		=>	'rss',
			'fa_style'		=>	'fas',
			'fa_color'		=>	'#f26522',
		),
		'tumblr'		=> 	array(
			'id'			=>	'tumblr',
			'label'			=>	'Tumblr',
			'image'			=>	null,
			'fa_icon'		=>	'tumblr',
			'fa_style'		=>	'fab',
			'fa_color'		=>	'#32506d',
		),
		'twitter'		=> 	array(
			'id'			=>	'twitter',
			'label'			=>	'Twitter',
			'image'			=>	null,
			'fa_icon'		=>	'twitter',
			'fa_style'		=>	'fab',
			'fa_color'		=>	'#55acee',
		),
		'vimeo'			=> 	array(
			'id'			=>	'vimeo',
			'label'			=>	'Vimeo',
			'image'			=>	null,
			'fa_icon'		=>	'vimeo',
			'fa_style'		=>	'fab',
			'fa_color'		=>	'#1ab7ea',
		),
		'yahoo'			=> 	array(
			'id'			=>	'yahoo',
			'label'			=>	'Yahoo',
			'image'			=>	null,
			'fa_icon'		=>	'yahoo',
			'fa_style'		=>	'fab',
			'fa_color'		=>	'#410093',
		),
		'yelp'			=> 	array(
			'id'			=>	'yelp',
			'label'			=>	'Yelp',
			'image'			=>	null,
			'fa_icon'		=>	'yelp',
			'fa_style'		=>	'fab',
			'fa_color'		=>	'#af0606',
		),
		'youtube'		=> 	array(
			'id'			=>	'youtube',
			'label'			=>	'YouTube',
			'image'			=>	null,
			'fa_icon'		=>	'youtube',
			'fa_style'		=>	'fab',
			'fa_color'		=>	'#cd201f',
		),
	);
	
	// Allow Devs to remove or add additional networks.
	$sources = apply_filters( 'bne_social_media_sources', $sources );

	// Return names only
	if( 'labels' == $request ) {
		$labels = array();
		foreach( $sources as $index => $source ) {
			$labels[$index] = $source['label'];
		}
		return $labels;
	}
	
	// Return index only
	if( 'index' == $request && $id ) {
		return $sources[$id];
	}
		
	return $sources;
}



/*
 * 	Contact Bar Buttons
 *
 *	@since 		v1.0
 *	@updated 	v2.2.1
 *
*/
if( !function_exists( 'bne_contact_bar' ) ) :
	function bne_contact_bar( $location = '', $color = '', $shape = 'round_square', $class = '', $size = '', $align = 'left', $target = '' ) {

		// Get theme set profiles
		$profiles = get_theme_mod( 'social_profiles' );
		if( !$profiles) {
			return;
		}

		// Get location
		if( $location == 'header' ) {
			$color = get_theme_mod( 'header_contact_bar_color', 'color' );
			$shape = get_theme_mod( 'header_contact_bar_shape', 'rounded' );
			$size = get_theme_mod( 'header_contact_bar_size', '25px' );
		}
		
		// Get Link Target
		if( !$target ) {
			$target = ( true == get_theme_mod( 'social_profiles_target' ) ) ? '_blank': '_self';
		}
		
		// Custom Size
		if( $size ) {
			// Set all parts to the numerical value of $size and remove any units (px, 5, em)
			$size = preg_replace( "/[^0-9]/", "", $size );
			$font_size = floor( $size / 1.7 ); // Reduce the font size by 1.6 of $size
		}
		
		// Classes
		$classes = array(
			'contact-bar-'.$location,
			'contact-bar-'.$shape,
			'contact-bar-'.$align,
			$class,
		);

		// Build Contact Bar
		$output = '<div class="bne-contact-bar '.implode(' ', $classes).' clearfix">';
			$output .= '<ul class="social-media-'.$color.'" style="--size: '.$size.'px; --font-size: '.$font_size.'px;">';
				
				// Loop through each profile
				foreach( $profiles as $profile ) {
					
					// Add the matching icon array from main sources to the requested property
					$profile['icon'] = bne_get_social_media_sources( 'index', $profile['source'] );

					// Only continue if there is a url defined...
					if( $profile['url'] ) {
						$fa_icon = $profile['icon']['fa_icon'];
						$fa_style = $profile['icon']['fa_style'];
						$icon_color = ( $color == 'color' ) ? 'background:'.$profile['icon']['fa_color'].';' : null;
						$icon_color .= ( $shape == 'transparent' ) ? 'color:'.$profile['icon']['fa_color'].';' : null;
						
						// Overrides
						if( 'email' == $profile['source'] ) {
							$url = 'mailto:'.sanitize_email( $profile['url'] );
						} else {
							$url = esc_url( $profile['url'] );
						}

						// Output Icon with link
						$output .= '<li class="'.$profile['icon']['id'].'" style="'.$icon_color.'">';
							$output .= '<a href="'.$url.'" title="'.$profile['icon']['label'].'" target="'.$target.'" aria-label="'.ucfirst( $profile['icon']['label'] ).'">';
								
								// Image Icon
								if( $profile['icon']['image'] ) {
									$output .= '<img src="'.esc_url( $profile['icon']['image'] ).'" class="bne-image aria-hidden="true" />';
								
								// FA Icon
								} else {
									$output .= '<i class="bne-fa '.$fa_style.' fa-fw fa-'.$fa_icon.'" aria-hidden="true"></i>';
								}
							
							$output .= '</a>';
						$output .= '</li>';
					}
					
				} // End loop
					
			$output .= '</ul>';
		$output .= '</div>';

		return $output;
	
	}
endif;