<?php
/*
 * 	Framework Setup functions
 *
 * 	@author		Kerry Kline
 * 	@copyright	Copyright (c) Kerry Kline
 * 	@link		http://www.bnecreative.com
 * 	@package 	BNE WordPress Framework
 *
*/

// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;




/*
 * 	Page Options Wrapper
 *
 *	An organized array of options that are found both in Theme Options
 *	and pages/post Custom Fields. This prevents errors for templates such
 *	as 404, search, and archive who wouldn't have $post meta data.
 *
 *	@since 		v1.0
 *	@updated 	v2.1.4
 *
*/
function bne_get_page_option( $key = '' ) {
	
	$options = array();
	//$page_id = is_home() ? get_queried_object_id() : get_the_id();
	
	// Theme Options
	$sidebar_override = get_post_meta( get_the_id(), 'bne_sidebar_layout', true );
	$sidebar_page_default = get_theme_mod( 'sidebar_layout_pages', 'right-sidebar' );		
	$sidebar_post_default = get_theme_mod( 'sidebar_layout_posts', 'right-sidebar' );
	$sidebar_archive_default = get_theme_mod( 'sidebar_layout_archive', 'right-sidebar' );
	$breadcrumbs_default = get_theme_mod( 'breadcrumbs', 'hide' );

	// Initial Set
	$options = array(
		'sidebar_layout'	=>	get_post_meta( get_the_id(), 'bne_sidebar_layout', true ),
		'breadcrumbs'		=>	get_post_meta( get_the_id(), 'bne_breadcrumbs', true ),
		'header_display'	=>	get_post_meta( get_the_id(), 'bne_header_display', true ) ?: 'default',
		'footer_display'	=>	get_post_meta( get_the_id(), 'bne_footer_display', true ) ?: 'default'
	);

	// WP Templates
	if( is_search() || is_archive() || is_404() || is_home() ) {
		$options['sidebar_layout'] = 'right-sidebar';
		$options['breadcrumbs'] = $breadcrumbs_default;
		$options['header_display'] = 'default';
		$options['footer_display'] = 'default';
	}
	
	// Archive / Blog Page
	if( is_archive() || is_home() ) {
		$options['sidebar_layout'] = $sidebar_archive_default;
		if( is_home() ) {
			$options['header_display'] = get_post_meta( get_queried_object_id(), 'bne_header_display', true ) ?: 'default';
			$options['footer_display'] = get_post_meta( get_queried_object_id(), 'bne_footer_display', true ) ?: 'default';
		}
	}

	// Re-configure any defaults to what is set in Theme Options
	if( 'default-sidebar' == $options['sidebar_layout'] || !$options['sidebar_layout'] ) {
		if( is_singular( 'page' ) ) {
			$options['sidebar_layout'] = $sidebar_page_default;
		} elseif( is_singular( 'post' ) ) {
			$options['sidebar_layout'] = $sidebar_post_default;
		} elseif( is_archive() ) {
			$options['sidebar_layout'] = $sidebar_archive_default;
		}
	}
	if( 'default' == $options['breadcrumbs'] || !$options['breadcrumbs'] ) {
		$options['breadcrumbs'] = $breadcrumbs_default;
	}

	// Dev Allow developers to adjust page options conditionally. 
	$options = apply_filters( 'bne_get_page_option', $options );
	
	//bne_printr($options);
	// If a key is requested, let's return only that value.
	if( $key ) {
		return $options[$key];
	// Else Return all options
	} else {
		return $options;
	}

}



/*
 * 	Set up theme defaults, settings, and register support for various WordPress features.
 *
 * 	Note that this function is hooked into the after_setup_theme hook, which runs
 * 	before the init hook. The init hook is too late for some features, such as indicating
 * 	support post thumbnails.
 *
 *	@since 		v1.0
 *	@updated 	v2.2.10
 *
*/
if( ! function_exists( 'bne_theme_setup' ) ) :
	function bne_theme_setup() {
		global $content_width;
		global $wp_embed;


		// Make theme available for translation
		load_theme_textdomain( 'bne', get_template_directory() . '/languages' );
		
		
		/*
		 * 	Set Content Width for WordPress
		 *
		 * 	Some elements need a width set for WordPress
		 *	to use in various areas such as post-thumbnails
		 *	and oembeds.
		 *
		 *	2500 Width is a high point for "today" and is used
		 *	if the theme design width if set to 100%.
		 *
		*/
		$theme_width = get_theme_mod( 'content_width', '1200px' );
		
		if( $theme_width != '100%' ) {
			$content_width = str_replace( array( 'px', '%' ), '', $theme_width );
		} else {
			$content_width = 2500;
		}
		
		// Set Post Thumbnail max size to the $content_width
		set_post_thumbnail_size( $content_width, 9999 );

		// Set Max srcset Image Width for Responsive Images (WP 4.4)
		add_filter( 'max_srcset_image_width', function() { return '2500'; } );

		// Add default posts and comments RSS feed links to head
		add_theme_support( 'automatic-feed-links' );

		// Enable support for Post Thumbnails on posts and pages
		add_theme_support( 'post-thumbnails' );
		
		// Jetpack Infinite Scroll
		add_theme_support( 'infinite-scroll', array( 'container' => 'content', 'footer' => 'wrapper' ) );

		// HTML5 Support
		add_theme_support( 'html5', array( 'search-form', 'comment-form', 'comment-list', 'gallery', 'caption' ) );
					
		// WP 4.1 'title-tag' Support that replaces the wp_title() added to header.php.
		add_theme_support( 'title-tag' );
		
		// Block Editor Support
		add_theme_support( 'wp-block-styles' );
		add_theme_support( 'align-wide' );
		add_theme_support( 'responsive-embeds' );
		add_theme_support( 'custom-units' );
		add_theme_support( 'editor-styles' );
		add_editor_style( BNE_FRAMEWORK_ASSETS . '/css/editor.css' );
		
		// WP Customizer
    	add_theme_support( 'custom-logo', array(
        	'height'      => 100,
			'width'       => 400,
			'flex-height' => true,
			'flex-width'  => true,
			'header-text' => array( 'site-title', 'site-description' ),
		) );

		// Enhance the WP Text Widgets to allow shortcodes and oembeds
		add_filter( 'widget_text', 'do_shortcode' );
		add_filter( 'widget_text', array( $wp_embed, 'run_shortcode' ), 8 );
		add_filter( 'widget_text', array( $wp_embed, 'autoembed'), 8 );
				
		// Register Theme Menu Loations
		register_nav_menus( array(
			'header-menu'  	=>	esc_html__( 'Header Menu', 'bne' ),
			'footer-menu'  	=>	esc_html__( 'Footer Menu', 'bne' ),
			'sticky-menu'	=>	esc_html__( 'Sticky Menu', 'bne' ),
			'mobile-menu'  	=>	esc_html__( 'Mobile Menu', 'bne' ),
		) );

		// Register Custom Color Palette
		$color_palette = apply_filters( 'bne_theme_color_palette', array(
			// Theme
			'primary' => array(
				'name'  => __( 'Primary', 'bne' ),
				'slug'  => 'primary',
				'color'	=> get_theme_mod( 'theme_primary_color', '#0099ff' ),
			),
			'secondary' => array(
				'name'  => __( 'Secondary', 'bne' ),
				'slug'  => 'secondary',
				'color'	=> get_theme_mod( 'theme_secondary_color', '#7892BC' ),
			),
			'tertiary' => array(
				'name'  => __( 'Tertiary', 'bne' ),
				'slug'  => 'tertiary',
				'color'	=> get_theme_mod( 'theme_tertiary_color', '#EEE8A9' ),
			),
			'dark' => array(
				'name'	=> __( 'Dark', 'bne' ),
				'slug'	=> 'dark',
				'color'	=> '#222',
			),
			'light' => array(
				'name'	=> __( 'Light', 'bne' ),
				'slug'	=> 'light',
				'color'	=> '#f5f5f5',
			),		
			// Bootstrap
			'success' => array(
				'name'  => __( 'Success', 'bne' ),
				'slug'  => 'success',
				'color' => '#28a745',
			),
			'danger' => array(
				'name'  => __( 'Danger', 'bne' ),
				'slug'  => 'danger',
				'color' => '#dc3545',
			),
			'warning' => array(
				'name'  => __( 'Warning', 'bne' ),
				'slug'  => 'warning',
				'color' => '#ffc107',
			),
			'info' => array(
				'name'  => __( 'Info', 'bne' ),
				'slug'  => 'info',
				'color' => '#17a2b8',
			),
		) );
		if( !empty($color_palette) ) {
			// @note using array_values to remove the array index name with WP 5.4
			add_theme_support( 'editor-color-palette', array_values($color_palette) );
			
			// build css
			$css = '';
			foreach( $color_palette as $color ) {
				// WP needs the slug to use hyphen and not underscores, let's check it
				$color['slug'] = str_replace('_', '-', $color['slug']);
				
				$css .= ".has-{$color['slug']}-color { color: {$color['color']}; }";
				$css .= ".has-{$color['slug']}-background-color { background-color: {$color['color']}; }";
			}
			$css .= ":root {";
			foreach( $color_palette as $color ) {
				// WP needs the slug to use hyphen and not underscores, let's check it
				$color['slug'] = str_replace('_', '-', $color['slug']);

				$css .= "--theme-{$color['slug']}-color: {$color['color']};";
			}
			$css .= "}";
			
			// Add to frontend
			add_action( 'wp_enqueue_scripts', function() use($css) {
				wp_add_inline_style( 'bne_theme_parent', $css );
			});
			
			// Add to admin editor (via bne-blocks)
			add_action( 'admin_enqueue_scripts', function() use($css) {
				
				// Default button color
				$button_color = get_theme_mod( 'button_color', '#0099ff' );
				$css .= '.wp-block-button__link { background-color: '.$button_color.'; }';
				wp_add_inline_style( 'bne-blocks', $css );
			});
		}

		// Get the Primary Menu Location
		$menu_location = get_theme_mod( 'menu_location', 'default' );

		// Above Header
		if( 'above_header' == $menu_location ) {
			add_action( 'bne_header_before', 'bne_header_menu_default' );

		// Opposite of Logo
		} elseif( 'header_addon' == $menu_location ) {
			add_action( 'bne_header_addon_after', 'bne_header_menu_default' );
		
		// Below Header
		} elseif( 'below_header' == $menu_location ) {
			add_action( 'bne_header_after', 'bne_header_menu_default' );
		
		// Default (as stated in header.php)
		} else {
			add_action( 'bne_header_menu', 'bne_header_menu_default' );
		}
		
		// Sticky Header/Menu
		if( 'true' == get_theme_mod( 'sticky_header' ) ) {
			add_action( 'bne_after', 'bne_header_sticky_default' );
		}

		// Header Sidebar Layout
		$site_layout = get_theme_mod( 'layout', 'wide_layout' );
		if( $site_layout == 'header_sidebar_layout' ) {
						
			// Header Content Widget
			add_action( 'bne_header_content_after', function() {
				bne_header_sidebar_content();
			});
			
			// Disable Transparent Header
			add_filter( 'bne_get_page_option', function( $options ) {
				if( 'transparent' == $options['header_display'] ) {
					$options['header_display'] = 'default';
				}
				return $options;
			});
			
			// Menu Location Overrides
			remove_action( 'bne_header_before', 'bne_header_menu_default' );
			remove_action( 'bne_header_addon_after', 'bne_header_menu_default' );
			remove_action( 'bne_header_after', 'bne_header_menu_default' );
			add_action( 'bne_header_menu', 'bne_header_menu_default' );

		}
			
	}
endif;
add_action( 'after_setup_theme', 'bne_theme_setup' );