<?php
/*
 * 	Theme and Framework Structure including output of most areas.
 *
 * 	@author		Kerry Kline
 * 	@copyright	Copyright (c) Kerry Kline
 * 	@link		http://www.bnecreative.com
 * 	@package 	BNE WordPress Framework
 *
*/

// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;



/*
 * 	Header Content
 *
 *	Sets up the main header left/right content areas
 *	such as the logo, message, phone, address, and social links.
 *  Can be altered for by moving this to a child theme.
 *
 *	@note		Found in header.php
 *
 *	@since 		v1.0
 *	@updated 	v1.5.15
 *
*/
if( !function_exists( 'bne_header_content' ) ) :
	function bne_header_content() {

		// Get Theme Options
		$header_layout = get_theme_mod( 'header_columns', 'half_half' );

		?>
		<div class="header-content header-<?php echo $header_layout; ?> clearfix">
			<div class="row grid-protection">

				<div class="<?php echo bne_get_column_class( 'left-header', 'header' ); ?> logo-col clearfix">
					<?php do_action( 'bne_header_logo_before' ); ?>
					<?php bne_header_logo( 'header' ); ?>
					<?php do_action( 'bne_header_logo_after' ); ?>
				</div><!-- Left Header Column (end) -->

				<div class="<?php echo bne_get_column_class( 'right-header', 'header' ); ?> content-col clearfix">
					<?php echo do_action( 'bne_header_addon_before' ); ?>
					<?php echo bne_header_addon(); ?>
					<?php echo do_action( 'bne_header_addon_after' ); ?>
				</div><!-- Right Header Column (end) -->

			</div><!-- .row (end) -->
		</div><!-- .header-content (end) -->
		<?php
	}
endif;



/*
 * 	Header Header Add-on
 *
 *	The content specifically associated to the opposite side
 *	of the site logo.
 *
 *	@since 		v1.5.9
 *	@updated	v2.0
 *
*/
if( !function_exists( 'bne_header_addon' ) ) :
	function bne_header_addon() {
		$site_layout = get_theme_mod( 'layout', 'wide_layout' );
		$contact_bar = get_theme_mod( 'header_contact_bar', true );
		
		if( is_active_sidebar( 'header-sidebar' ) || $contact_bar == 'false' ) {
			echo '<div class="header-addon clearfix">';
				// Social Icons
				if( $contact_bar == 'true' ) {
					echo bne_contact_bar( 'header' );
				}
				// Header Widget (normal header layout)
				if( $site_layout != 'header_sidebar_layout' ) {
					echo '<div class="header-widget-area">';
						if( !dynamic_sidebar( 'header-sidebar' ) );
					echo '</div>';
				}
			echo '</div>';
		}
	}
endif;



/*
 * 	Header Sidebar Content
 *
 *	If using the Header Sidebar Layout, this will output
 *	a widget area below the primary menu.
 *
 *	@since 		v1.6
 *	@updated	v2.0
 *
*/
if( !function_exists( 'bne_header_sidebar_content' ) ) :
	function bne_header_sidebar_content() {
		echo '<div class="header-sidebar-widgets">';
			if( !dynamic_sidebar( 'header-sidebar' ) );
		echo '</div>';
	}
endif;



/*
 * 	Logo
 *
 *	Configures the output of the website logo and non logo text.
 *  Can be altered by moving this to a child theme.
 *
 *	@since 		v1.0
 *	@updated 	v2.2.9
 *
*/
if( !function_exists( 'bne_header_logo' ) ) :
	function bne_header_logo( $location = 'header' ) {

		// Logo
		$default_logo = null;
		
		// WP Customizer Logo
		if( has_custom_logo() ) {
			$wp_logo = wp_get_attachment_image_src( get_theme_mod( 'custom_logo' ) , 'full' );
			$default_logo = $wp_logo[0];
		}
		
		// Alternate Logos
		$overlay_logo = esc_url( get_theme_mod( 'logo_overlay' ) );
		$sticky_logo = esc_url( get_theme_mod( 'logo_sticky' ) );
		$mobile_logo = esc_url( get_theme_mod( 'logo_mobile' ) );
		

		// Logo Classes
		$classes = 'header-logo';
		if( $default_logo ) { $classes .= ' has-default-logo'; }
		if( $mobile_logo ) { $classes .= ' has-mobile-logo'; }
		if( $overlay_logo && 'transparent' == bne_get_page_option( 'header_display' ) ) { $classes .= ' has-overlay-logo'; }

		?>

		<div class="<?php echo $classes; ?>">
			<a href="<?php echo home_url(); ?>" title="<?php echo get_bloginfo('name'); ?>">
				<?php
					// Header Logos
					if( $location == 'header' ) {
						if( $default_logo ) {
							echo '<img src="'.$default_logo.'" class="default-logo" alt="'.get_bloginfo('name').'"/>';
							if( $mobile_logo ) {
								echo '<img src="'.$mobile_logo.'" class="mobile-logo" alt="'.get_bloginfo('name').'"/>';
							}
						}
						
						elseif( get_theme_mod( 'header_text', true ) ) {
							echo '<span class="site-title">'.get_bloginfo( 'name' ).'</span>';
							echo '<span class="site-description">'.get_bloginfo( 'description' ).'</span>';
						}
						
						if( $overlay_logo ) {
							echo '<img src="'.$overlay_logo.'" class="overlay-logo" alt="'.get_bloginfo('name').'"/>';
						}
					}
					
					// Sticky Logo
					if( $location == 'sticky' ) {
						if( $sticky_logo ) {
							echo '<img src="'.$sticky_logo.'" class="sticky-logo" alt="'.get_bloginfo('name').'"/>';
						} elseif( $default_logo ) {
							echo '<img src="'.$default_logo.'" class="sticky-logo" alt="'.get_bloginfo('name').'"/>';
						} elseif( get_theme_mod( 'header_text' ) ) {
							echo '<span class="site-title">'.get_bloginfo( 'name' ).'</span>';
						}
					}
				?>
			</a>
		</div><!-- .header-logo (end) -->

		<?php
	}
endif; // bne_header_logo



/*
 * 	Primary Menu
 *
 *	Configures the output of the primary menu. Includes
 *	additional hooks for custom content. Can be altered
 *	for by moving this to a child theme.
 *
 *	@since 		v1.0
 *	@updated 	v1.5.9
 *
*/
if( !function_exists( 'bne_header_menu_default' ) ) :
	function bne_header_menu_default() {
		
		$menu_class = 'menu-location-'.get_theme_mod( 'menu_location', 'default' );
		$menu_location = apply_filters( 'bne_header_menu_location', 'header-menu' );
		$has_mobile_menu = ( has_nav_menu( 'mobile-menu' ) ) ? 'has-mobile-menu' : '';
		$menu_extras = get_theme_mod( 'menu_extras' );
		
		do_action( 'bne_header_menu_before' );

		?>
		<!-- Main Menu -->
		<div class="menu-wrapper mobile-<?php echo get_theme_mod( 'responsive_menu', 'default' ); ?> <?php echo $has_mobile_menu; ?> clearfix">
			
			<?php echo bne_mobile_nav_btn('close'); ?>
			
			<nav id="access" class="header-menu <?php echo $menu_class; ?>" role="navigation" itemscope itemtype="http://www.schema.org/SiteNavigationElement">
				<div class="access-inner content-area-width">
					<div class="access-content clearfix">
						
						<?php wp_nav_menu( array( 'menu_id' => 'primary-menu', 'menu_class' => 'sf-menu standard-menu', 'container' => '', 'theme_location' => $menu_location, 'fallback_cb' => false ) ); ?>
						
						<?php wp_nav_menu( array( 'menu_id' => 'primary-menu', 'menu_class' => 'sf-menu mobile-menu', 'container' => '', 'theme_location' => 'mobile-menu', 'fallback_cb' => false ) ); ?>
						
						<?php bne_header_menu_addon(); ?>						
					
					</div><!-- .access-content (end) -->
				</div><!-- .access-inner (end) -->
			</nav><!-- #access (end) -->
			
			<div class="off-canvas-menu-extras">
				<?php do_action( 'bne_side_menu_addon' ); ?>
			</div>
			
		</div><!-- .menu-wrapper (end) -->
		<?php

		do_action( 'bne_header_menu_after' );

	}
endif;



/*
 * 	Primary Menu Mobile Nav Button
 *
 *	Configures the output of the mobile nav button.
 *
 *	@since 		v1.4.2
 *	@updated	v2.0
 *
*/
function bne_mobile_nav_btn( $state = 'open' ) {
	$label = bne_dynamic_variables( get_theme_mod( 'mobile_nav_label' ) ) ?: '<i class="bne-fa fas fa-fw fa-bars"></i>';
	
	if( 'close' == $state ) {
		?>
		<!-- Side Nav Close Menu Button -->
		<a href="#access" class="btn-navbar primary-menu-toggle close-nav" aria-controls="primary-menu" aria-expanded="false" aria-label="<?php _e('Close Mobile Menu', 'bne'); ?>">
			<span><?php echo __('Close', 'bne'); ?></span> <i class="bne-fa fas fa-fw fa-times"></i>
		</a>
		<?php 
	
	} else { 
		?>
		<!-- Mobile Menu Button -->
		<a href="#access" class="btn-navbar primary-menu-toggle open-nav" aria-controls="primary-menu" aria-expanded="false" aria-label="<?php _e('Open Mobile Menu', 'bne'); ?>">
			<?php echo apply_filters( 'bne_btn_navbar_text', $label ); ?>
		</a>
		<?php 
	}	
	
}
add_action( 'bne_header_before', 'bne_mobile_nav_btn' );



/*
 * 	Sticky Header
 *
 *	A duplication of the header with only the logo
 *	and primary menu. Includes additional hooks for
 *	custom content. Can be altered for by moving this
 *	to a child theme.
 *
 *	@since 		v1.2.0
 *	@updated	v2.0
 *
*/
if( !function_exists( 'bne_header_sticky_default' ) ) :
	function bne_header_sticky_default() {
		$menu_location = ( has_nav_menu( 'sticky-menu' ) ) ? 'sticky-menu' : 'header-menu';
		$menu_location = apply_filters( 'bne_sticky_menu_location', $menu_location );
		?>

		<!-- Sticky Main Menu -->
		<div id="sticky-header" class="bne-sticky-menu">

			<?php do_action( 'bne_header_sticky_before' ); ?>

			<div class="content-area-width clearfix">

				<!-- Nav -->
				<div class="nav">
					<?php wp_nav_menu( array( 'menu_id' => 'primary-menu', 'menu_class' => 'sf-menu', 'container' => '', 'theme_location' => $menu_location, 'fallback_cb' => false ) ); ?>
					<?php bne_header_menu_addon(); ?>
				</div>

				<!-- Logo -->
				<?php bne_header_logo( 'sticky' ); ?>

			</div>

			<?php do_action( 'bne_header_sticky_after' ); ?>

		</div><!-- #sticky-menu (end) -->

		<?php
	}
endif;



/*
 * 	Extra Menu Links
 *
 *	Uses the wp_nav_menu_items filter to include additional menu items
 *	outside the WP admin Menu page.
 *
 *	$param	$items	The HTML list content for the menu items.
 *	$param	$args	An object containing wp_nav_menu() arguments.
 *
 *	@since 		v1.4.2
 *	@updated 	v2.0
 *
*/
function bne_menu_extras( $items, $args ) {
	
	// Get Menu Options
	$menu_extras = get_theme_mod( 'menu_extras' );
	if( !$menu_extras ) {
		return $items;
	}
	
	// Only add to the primary menu
	if( $args->theme_location == 'header-menu' ) {

		// Search Link
		if( in_array('search', $menu_extras ) ) {
			$items .= '<li class="menu-item menu-search">';
				$items .= '<a href="#" data-toggle="modal" data-target="#modal_search"><i class="bne-fa fas fa-search"></i> <span class="hide_on_standard_and_tablet">'.esc_html__( 'Search', 'bne' ).'</span></a>';
			$items .= '</li>';
			// Output the modal at the end of the document.
			add_action( 'bne_after', 'bne_search_modal' );
		}
	
		// WooCommerce Cart
		if( class_exists( 'WooCommerce' ) && in_array( 'cart', $menu_extras ) ) {
			// Cart Quantity
			$qty = WC()->cart->get_cart_contents_count();
			
			// Configure Button action (modal)
			if( !is_cart() && !is_checkout() ) {
				$modal = 'data-toggle="modal" data-target="#modal_cart"';
				$active = '';
			} else {
				$modal = 'class="no-click"';
				$active = 'current_page_item';
			}
			
			// Shopping Cart Button
			$items .= '<li class="menu-item menu-cart '.$active.'">';
				$items .= '<a href="#" '.$modal.'>';
					$items .= '<i class="bne-fa fas fa-shopping-cart"></i><span class="badge">'.$qty.'</span> ';
					$items .= '<span class="hide_on_standard_and_tablet">'.esc_html__( 'Shopping Cart', 'bne' ).'</span>';
				$items .= '</a>';
			$items .= '</li>';
			
			// Output the modal at the end of the document.
			add_action( 'bne_after', 'bne_cart_modal' );
		
		} // End Woo check
	
		

	} // End header-menu check

	// Return New Menu Items
	return $items;
	
}
add_filter( 'wp_nav_menu_items', 'bne_menu_extras', 10, 2 );



/*
 * 	Shopping Cart Modal
 *
 *	A global modal (bootstrap) that display the shopping cart
 *	for WooCommerce. Can be altered for other carts by moving
 *	this to a child theme.
 *
 *	@since 		v1.4.2
 *
*/
if( !function_exists( 'bne_cart_modal' ) ) :
	function bne_cart_modal() {
		?>
		<div id="modal_cart" class="modal fade bne-cart-popup" tabindex="-1" role="dialog">
			<div class="modal-dialog modal-sm">
				<div class="modal-content">
					<div class="modal-header">
						<h5 class="modal-title"><?php _e( 'Shopping Cart', 'bne' ); ?></h5>
						<button type="button" class="close" data-dismiss="modal" aria-label="Close"><span aria-hidden="true">&times;</span></button>
					</div>
					<div class="modal-body">
						<?php woocommerce_mini_cart(); ?>
					</div>
				</div>
			</div>
		</div>
		<?php	
	}
endif;



/*
 * 	Search Modal
 *
 *	A global modal (bootstrap) that display the search field
 *	Can be altered for other carts by moving this to a child theme.
 *
 *	@since 		v1.4.2
 *
*/
if( !function_exists( 'bne_search_modal' ) ) :
	function bne_search_modal() {
		?>
		<div id="modal_search" class="modal fade bne-search-popup" tabindex="-1" role="dialog">
			<div class="modal-dialog modal-lg">
				<div class="modal-content">
					<div class="modal-header">
						<button type="button" class="close" data-dismiss="modal" aria-label="Close">X</button>
					</div>
					<div class="modal-body">
						<?php get_search_form(); ?>
					</div>
				</div>
			</div>
		</div>	
		<?php	
	}
endif;



/*
 * 	Featured Area
 *
 *	Sets up the conditional logic of displaying the featured
 *	area between the header and the main wrappers. Can be
 *	altered by moving this to a child theme.
 *
 *	Primarily used to display the featured image on pages
 *	and for the transparent.
 *
 *	@since 		v1.0
 *	@updated	v2.2.7
 *
*/
if( !function_exists( 'bne_featured_area_default' ) ) :
	function bne_featured_area_default() {
		
		// Kill this area if the image is used in bne_page_heading_wrapper()
		$is_page_heading_image = get_post_meta( get_the_id(), 'bne_page_heading_bg', true );
		if( 'true' == $is_page_heading_image ) {
			return;
		}

		$page_id = is_home() ? get_queried_object_id() : get_the_id();
		$post_types = array('page');
		if( is_single() ) {
			$header_option = get_post_meta( $page_id, 'bne_header_display', true );
			if( $header_option == 'transparent' ) {
				$post_types[] = 'post';
			}
		}

		// Page Featured Image
		if( ( is_singular( $post_types ) || is_home() ) && has_post_thumbnail() ) {
			echo '<div id="featured-wrapper" class="full-width-featured-area page-featured-image">';
				echo '<div class="content-area-width">';
					echo get_the_post_thumbnail( $page_id, 'full' );
					echo '<div class="content-shadow shadow-type'.apply_filters( 'bne_featured_area_shadow', '0' ).'"></div>';
				echo '</div><!-- .content-area-width (end) -->';
			echo '</div><!-- #featured-wrapper (end) -->';
		}
	}
endif;






/*
 *	Page Heading Wrapper
 *
 *	Wraps the page heading to allow replacement outside
 *	the main article.
 *
 *	@since v2.2.7
 *
*/
if( !function_exists( 'bne_page_heading_wrapper' ) ) :
	function bne_page_heading_wrapper() {
		if( is_front_page() ) {
			return;
		}
		
		// Get Settings
		$display = get_post_meta( get_the_id(), 'bne_title', true );
		$page_heading_bg = get_theme_mod( 'page_heading_bg' );
		$text_color = get_post_meta( get_the_id(), 'bne_page_heading_color', true );

		
		// Skip if we are not showing the page title
		if( 'hide' == $display ) {
			return;
		}

		
		// Remove default Breadcrumb and heading locations
		remove_action( 'bne_breadcrumbs', 'bne_breadcrumbs_default' );
		remove_action( 'bne_page_heading', 'bne_page_heading_default' );

		
		// Allow pages to use the featured image as the bg
		if( is_singular( array( 'page' ) ) && has_post_thumbnail() ) {
			$use_featured_image = get_post_meta( get_the_id(), 'bne_page_heading_bg', true );
			if( 'true' == $use_featured_image ) {
				$page_heading_bg['background-image'] = get_the_post_thumbnail_url( get_the_id(), 'full' );
			}
		}
		
		// Classes
		$classes = 'text-'.$display.' text-'.$text_color;
		if( $page_heading_bg['background-image'] ) {
			$classes .= ' has-background-image';
		}
		
		
		echo '<div id="page-heading-wrapper" class="'.$classes.'" style="'.bne_generate_css_properties( $page_heading_bg ).'">';
			echo '<div class="content-area-width">';
				bne_page_heading_default();
				bne_breadcrumbs_default();
			echo '</div>';
		echo '</div>';
	
	}
endif;
if( get_theme_mod( 'page_heading', 'default') == 'featured' ) {
	add_action( 'bne_main_before', 'bne_page_heading_wrapper' );
}



/*
 * 	Main Content Area Start
 *
 *	The opening wrapper of the main content area. Can be
 *	altered for by moving this to a child theme.
 *
 *	@since 		v1.0
 *
*/
if( !function_exists( 'bne_main_start' ) ) :
	function bne_main_start() {
		?>
		<div id="main-content-wrapper">
			<div class="main-inner">
				<div class="content-area-width">
		<?php
	}
endif;



/*
 * 	Main Content End
 *
 *	The closing wrapper of the main content area. Can be
 *	altered for by moving this to a child theme.
 *
 *	@since 		v1.0
 *
*/
if( !function_exists( 'bne_main_end' ) ) :
	function bne_main_end() {

		?>
				</div><!-- .content-area-width (end) -->
			</div><!-- .main-inner (end) -->
		</div> <!-- #main-content-wrapper (end) -->
		<?php
	}
endif;



/*
 * 	Sidebar Logic
 *
 *	Determines which sidebar to display based on the $side
 *	called from.
 *
 *	Notes		To set a sidebar layout for a custom post type
 *				use bne_get_page_option() filter and assign
 *				the correct sidebar_layout value.
 *				Filter: /framework/includes/setup.php:81
 *
 *				To set a widget area of a custom post type
 *				use bne_get_left_sidebar_widget_area and
 *				bne_get_right_sidebar_widget_area with the widget
 *				area slug.
 *				Filters: /sidebar-left.php & /sidebar-right.php
 *
 *	@param		$side		The sidebar location from the template
 *							either left or right
 *
 *	@since 		v1.0
 *	@updated 	v1.4.2
 *
*/
function bne_sidebars( $side ) {

	// Get Sidebar Layout
	$sidebar_layout = bne_get_page_option( 'sidebar_layout' );
		
	// If Left Sidebar
	if( 'left-sidebar' == $sidebar_layout && 'left' == $side ) {
		get_sidebar( 'left' );
	}
	
	// If Right Sidebar
	elseif( 'right-sidebar' == $sidebar_layout && 'right' == $side ) {
		get_sidebar( 'right' );
	}

	// If Double Sidebar
	elseif( 'double-sidebar' == $sidebar_layout ) {
		if( 'left' == $side ) {
			get_sidebar( 'left' );
		} else {
			get_sidebar( 'right' );
		}
		
	}

}



/*
 * 	Footer Content - Footer Widgets
 *
 *	Sets up the conditional logic of displaying the footer
 *	widgets within the footer wrapper. Can be
 *	altered for by moving this to a child theme.
 *
 *	@since 		v1.0
 *	@updated	v2.0
 *
*/
if( !function_exists( 'bne_footer_content_default' ) ) :
	function bne_footer_content_default() {

		// Get Theme Option Footer Setup
		$footer_setup = get_theme_mod( 'footer_columns', 0 );

		//Check first if columns are enabled!
		if( $footer_setup > '0' ) {

			/*
			 * 	Based on the column option, we need to first setup
			 *	the pre v1.4.2 columns. These values to do not include
			 *	the bootstrap grid patter so we need to define it here.
			*/
			if( $footer_setup == '1' ) { $footer_setup = '12'; }
			elseif( $footer_setup == '2' ) { $footer_setup = '6-6'; }
			elseif( $footer_setup == '3' ) { $footer_setup = '4-4-4'; }
			elseif( $footer_setup == '4' ) { $footer_setup = '3-3-3-3'; }
			elseif( $footer_setup == '5' ) { $footer_setup = '15-15-15-15-15'; } // "15" is "one-fifth"
			/*
			 *	Still here... must be using the new complex column layouts.
			 *	The complex layouts begin with a "#_" that represents
			 *	the number of columns in the row. We can skip that part here
			 *	and start on the 3rd character of the option value. This value
			 *	already defines the bootstrap grid pattern.
			*/
			else { $footer_setup = substr( $footer_setup, 2 ); }
			
			// Start a column count
			$i = 1;

			// Break up the column pattern into an array
			$col_pattern = explode('-', $footer_setup);
			
			// Open footer widget area
			echo '<div id="footer_content" class="footer-widgets">';
				echo '<div class="row grid-protection">';
					
					/*
					 * 	Footer Column Loop
					 *
					 * 	Builds each footer column. We start with 1 and add another until
					 *	we meet the pattern total.
					 *	
					*/			 
					foreach( $col_pattern as $col ) {
						
						// Set Footer Column ID
						$footer_id = 'footer-column'.$i.'-widget-area';
						
						// Build each footer column
						echo '<div id="'.$footer_id.'" class="footer-column col-md-'.$col.' clearfix">';
	
							if( !dynamic_sidebar( $footer_id ) ) {
								if( current_user_can( 'edit_theme_options' ) ) {
									echo '<div class="alert alert-warning">'.esc_html__( "You haven't setup any widgets yet for this widget area.", 'bne' ).'</div>';
								}
							}
	
						echo '</div><!-- .footer-column (end) -->';
	
						// Increase the count
						++$i;
	
					}
				
				echo '</div><!-- .row (end) -->';
			echo '</div><!-- .footer-widgets (end) -->';

		} // End Column Markup

	}
endif;



/*
 *	Alternate Footer Widgets Location
 *
 *	Adds the footer widgets above the Footer Copy in it's
 *	own wrapper.
 *
 *	@updated	v2.2.13
*/
if( !function_exists( 'bne_footer_widgets_alternate' ) ) :
	function bne_footer_widgets_alternate() {
		
		// Check if the page is hiding the footer
		if( 'hide' != bne_get_page_option( 'footer_display' ) ) {
			?>
			<div id="footer-widgets-wrapper">
				<div class="content-area-width">
					<?php echo bne_footer_content_default(); ?>
				</div>
			</div>
		<?php
		}
	}
endif;



/*
 * 	Footer Sub Content - Footer Copy/Footer Menu
 *
 *	Displays the bottom of the website with the
 *	site copy and footer menu. Can be altered for
 *	by moving this to a child theme.
 *
 *	@since 		v1.0
 *	@updated	v2.2.5
 *
*/
if( !function_exists( 'bne_footer_sub_content' ) ) :
	function bne_footer_sub_content() {
		
		$menu_location = apply_filters( 'bne_footer_menu_location', 'footer-menu' );
		$footer_layout = get_theme_mod( 'footer_sub_content_layout', 'default' );
		$col_size = ( $footer_layout == 'default' ) ? '6' : '12';
		
		do_action( 'bne_footer_sub_content_before' ); 
		
		?>
		<div id="footer_sub_content" class="footer-copy footer-layout-<?php echo $footer_layout; ?>">
			<div class="row grid-protection">

				<!-- Copy -->
				<div class="col-md-<?php echo $col_size; ?>">
					<div class="copyright">
						<div class="copyright-inner">
							<?php echo do_shortcode( bne_dynamic_variables( get_theme_mod( 'footer_copytext', '{icon_copyright} {year} {site_title}' ) ) ); ?>
						</div>
					</div><!-- .copyright (end) -->
				</div><!-- .col-sm (end) -->

				<!-- Footer Menu -->
				<div class="col-md-<?php echo $col_size; ?>">
					<div class="footer-nav">
						<div class="footer-inner">
							<?php wp_nav_menu( array( 'menu_id' => 'footer-menu', 'container' => '', 'fallback_cb' => false, 'theme_location' => $menu_location, 'depth' => 1 ) ); ?>
						</div>
					</div><!-- .footer-nav (end) -->
				</div><!-- .col-sm (end) -->

			</div><!-- .row (end) -->
		</div><!-- .footer-copy (end) -->

		<?php
			
		do_action( 'bne_footer_sub_content_after' ); 

	}
endif;