<?php
/*
 * 	BNE Block Editor Functions
 *
 * 	@author		Kerry Kline
 * 	@copyright	Copyright (c) Kerry Kline
 * 	@link		http://www.bnecreative.com
 * 	@package 	BNE WordPress Framework
*/

// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;




/*
 *	Add Dark Theme Editor body class
 *
 *	@since 		v2.2.8
 *	@updated	v2.2.12
 *
*/
function bne_block_admin_classes( $classes ) {
	//if( true == get_theme_mod( 'dark_theme' ) ) {
	if( true == apply_filters( 'bne_editor_dark_theme_mode', false ) ) {
		$classes = ' bne-editor-dark-theme';
	}
	return $classes;
}
add_filter( 'admin_body_class', 'bne_block_admin_classes' );




/*
 *	Register Block categories
 *
 *	@since 		v2.1
 *
*/
function bne_block_categories( $categories, $post ) {
	$slug = array_search( 'bne-blocks', array_column( $categories, 'slug' ) );
	if( !$slug ) { 
		$categories = array_merge(
			$categories,
			array( 
				array(
					'slug' => 'bne-blocks',
					'title' => __( 'BNE Blocks', 'bne' ),
				)
			)	
		);
	}
	
	return $categories;
	
}
if ( version_compare( $GLOBALS['wp_version'], '5.8', '<' ) ) {
	add_filter( 'block_categories', 'bne_block_categories', 10, 2 );
} else {
	// WP 5.8+
	add_filter( 'block_categories_all', 'bne_block_categories', 10, 2 );
}




/*
 *	Enqueue Block Editor Scripts
 *
 *	@since 		v2.1
 *	@updated	v2.2.8
 *
*/
function bne_block_editor_scripts() {
	global $pagenow;
	if ( $pagenow === 'widgets.php' ) {
		$editor_dependencies = array( 'wp-blocks', 'wp-components', 'wp-element', 'wp-i18n', 'wp-edit-widgets' );
	} else {
		$editor_dependencies = array( 'wp-blocks', 'wp-components', 'wp-element', 'wp-i18n', 'wp-editor' );
	}
	
	// Paths
	$block_path_uri = BNE_FRAMEWORK_ASSETS . '/blocks';
	$block_path_dir = BNE_FRAMEWORK_DIR . '/assets/blocks';

	// Styles
	wp_enqueue_style(
		'bne-blocks',
		$block_path_uri . '/bne-blocks.css',
		array( 'wp-edit-blocks' ),
		filemtime( $block_path_dir . '/bne-blocks.css' )
	);
	
	// FA Icons
	bne_enqueue_fontawesome();
		
	// General editor JS
	wp_enqueue_script(
		'bne-blocks',
		$block_path_uri . '/bne-blocks.js',
		$editor_dependencies,
		filemtime( $block_path_dir . '/bne-blocks.js' ),
		true
	);
	
	// Theme Blocks
	$blocks = apply_filters( 'bne_theme_blocks', array(
		'visibility', // First because of the use of addFilter()
		'alert',
		'clear',
		'callout',
		'card',
		'container',
		'fa-icon',
		'google-fonts',
		'list',
		'heading',
		'map',
		'modal',
		'posts',
		'shape-divider',
		'progress-bar',
		'toggle',
	) );
	foreach( $blocks as $block ) {
		wp_enqueue_script(
			'bne-blocks-'.$block,
			$block_path_uri . '/bne-blocks-'.$block.'.js',
			$editor_dependencies,
			filemtime( $block_path_dir . '/bne-blocks-'.$block.'.js' ),
			true
		);		
	}

	
	// Map Vars
    wp_localize_script(
        'bne-blocks-map',
        'bne_blocks_map_vars',
        array(
            'apiKey' => get_option( 'bne_block_map_apikey' ),
        )
    );
    
    
    // Get categories for post block
    $categories = get_categories();
    $category_list = array(
	    array(
		    'value'	=>	'',
		    'label'	=>	__( 'All Categories', 'bne' )
	    )
    );
    foreach( $categories as $category ) {
	    $category_list[] = array(
		    'value'	=>	$category->term_id,
		    'label'	=>	$category->name
	    );
    }

    // Post Vars
    wp_localize_script(
        'bne-blocks-posts',
        'bne_blocks_posts_vars',
        array(
	        'categories'	=>	$category_list,
            'siteURL' 		=> 	get_site_url(),
        )
    );

   
    // Google Font Vars
	$fontsJson = wp_remote_get( esc_url_raw( BNE_FRAMEWORK_ASSETS.'/fonts/fonts.json' ) );
	$fontsJson = json_decode( wp_remote_retrieve_body( $fontsJson ), true );
    wp_localize_script( 
    	'bne-blocks-google-fonts', 
    	'bne_blocks_google_fonts_vars', 
    	array( 
    		'themeAssets' => BNE_FRAMEWORK_ASSETS,
    		'fontsJson'	=>	$fontsJson
    	) 
    );
	
	// Editor Dark Theme Styles
	// Adjust the editor to match the front end when using
	// a dark style theme.
	$editor_css = '';
	//if( true == get_theme_mod( 'dark_theme' ) ) {
	if( true == apply_filters( 'bne_editor_dark_theme_mode', false ) ) {
		
		
		// Get the body background style
		$body_bg = get_theme_mod( 'body_bg' );
		if( $body_bg ) {
			$editor_css .= '.bne-editor-dark-theme .editor-styles-wrapper { '.bne_generate_css_properties( $body_bg ).' }';
		}
		
		// Body Font
		$body_text_color = get_theme_mod( 'body_font' );
		if( $body_text_color ) {
			$editor_css .= '.bne-editor-dark-theme .block-editor .editor-styles-wrapper { color: '.$body_text_color['color'].'; }';
		}
		
		// Links	
		$link_color = get_theme_mod( 'link_color', '' );
		if( $link_color ) {
			$editor_css .= '.bne-editor-dark-theme .block-editor .editor-styles-wrapper a { color: '.$link_color.'; }';
		}
		
		// Headings
		$heading_text_color = get_theme_mod( 'heading_font' );
		if( $heading_text_color ) {
			$editor_css .= '.bne-editor-dark-theme .editor-post-title .editor-post-title__input,
				.bne-editor-dark-theme .editor-styles-wrapper .editor-post-title__input::placeholder,
				.bne-editor-dark-theme .block-editor-default-block-appender textarea.block-editor-default-block-appender__content,
				.bne-editor-dark-theme .block-editor .editor-styles-wrapper h1,
				.bne-editor-dark-theme .block-editor .editor-styles-wrapper h2,
				.bne-editor-dark-theme .editor-styles-wrapper h3, 
				.bne-editor-dark-theme .editor-styles-wrapper h4, 
				.bne-editor-dark-theme .editor-styles-wrapper h5, 
				.bne-editor-dark-theme .editor-styles-wrapper h6 { color: '.$heading_text_color['color'].'; }
			';
		}

	}
	
	// Add the CSS after 'bne-blocks' stylesheet
	if( $editor_css ) {
		wp_add_inline_style( 'bne-blocks', $editor_css );
	}

}
add_action( 'enqueue_block_editor_assets', 'bne_block_editor_scripts' );




/*
 *	Block Render: core/button
 *	Filters into the block output to adjust
 *	the html.
 *
 *	@since 		v2.2.3
 *	@updated	v2.2.9.1
 *
*/
function bne_render_block_button( $block_content, $block ) {
	
	global $wp_version;
		
	// Block: Button
	if( 'core/button' !== $block['blockName'] ) {
		return $block_content;
	}
	
	//bne_printr($block);

	// Open Vars
	$vars = '';
	$extra_class = 'bne-block-btn';
	
	// Get Text Color (palette)
	if( isset( $block['attrs']['textColor'] ) ) {
		$vars .= '--text-color: var(--theme-'.$block['attrs']['textColor'].'-color);';
	
	// Get Text Custom Color
	} elseif( isset( $block['attrs']['style']['color']['text'] ) ) {
		$vars .= '--text-color: '.$block['attrs']['style']['color']['text'].';';
		
	// Use Default
	} else {
		// check for classes
		if( isset( $block['attrs']['className'] ) ) {
			// If outline btn
			if( strpos( $block['attrs']['className'], 'is-style-outline') !== false ) {
				$vars .= '--text-color: inherit;';
			} else {
				$extra_class .= ' has-default-text-color';
				$vars .= '--text-color: #fff;';
			}
		// Fallback
		} else {
			$extra_class .= ' has-default-text-color';
			$vars .= '--text-color: #fff;';
		}
	}


	// Get BG Color (palette)
	if( isset( $block['attrs']['backgroundColor'] ) ) {
		$vars .= '--bg-color: var(--theme-'.$block['attrs']['backgroundColor'].'-color);';

	// Get BG Custom Color
	} elseif( isset( $block['attrs']['style']['color']['background'] ) ) {
		$vars .= '--bg-color: '.$block['attrs']['style']['color']['background'].';';

	// Get BG Gradient Color (palette)
	} elseif( isset( $block['attrs']['gradient'] ) ) {
		$extra_class .= ' btn-gradient';
		$vars .= '--bg-color: inherit;';

	// Get BG Gradient Custom Color
	} elseif( isset( $block['attrs']['style']['color']['gradient'] ) ) {
		$extra_class .= ' btn-gradient btn-gradient-custom';
		$vars .= '--bg-color: '.$block['attrs']['style']['color']['gradient'].';';

	
	// Use default
	} else {
		$extra_class .= ' has-default-bg-color';
		$vars .= '--bg-color: var(--theme-default-btn-color);';
	}


	
	// Clear out any existing style tags as we will be using CSS var()
	$block_content = preg_replace( '/ style=("|\')(.*?)("|\')/','',$block_content );

	// Reapply border radius if available (legacy)
	if( isset( $block['attrs']['borderRadius'] ) ) {
		$vars .= 'border-radius:'.$block['attrs']['borderRadius'].'px;'; 
	} else if( isset( $block['attrs']['style']['border']['radius'] ) ) {
		$radius = $block['attrs']['style']['border']['radius'];
		// WP 5.9+
		if( version_compare( $wp_version, '5.8.9' , '>=' ) ) {
			if( is_array( $radius ) ) {
				$vars .= 'border-radius:'.implode(' ', $radius ); 
			} else {
				$vars .= 'border-radius:'.$radius; 
			}
		// WP 5.8
		} else {
			$vars .= 'border-radius:'.$radius.'px;'; 
		}
	}
	
	// Inject our new css variables and extra classes
	$block_content = str_replace( '<a class="wp-block-button__link', '<a style="'.$vars.'" class="wp-block-button__link '.$extra_class, $block_content );
	
	
	// Return new Button
	return $block_content;
}
add_filter( 'render_block', 'bne_render_block_button', 10, 2 );




/*
 *	Block Render: core/table
 *	Filters into the block output to adjust
 *	the html.
 *
 *	@since 		v2.2.3
 *
*/
function bne_render_block_table( $block_content, $block ) {
	
	// Block: Table
	if( 'core/table' !== $block['blockName'] ) {
		return $block_content;
	}

	
	// Inject basic bootstrap classes
	$block_content = str_replace( '<figure class="wp-block-table', '<figure class="wp-block-table table-responsive', $block_content );
	$block_content = str_replace( '<table>', '<table class="table">', $block_content );
	
	
	// Return new Button
	return $block_content;
}
add_filter( 'render_block', 'bne_render_block_table', 10, 2 );




/*
 *	Block Render: ALL (visibility options)
 *	Filters into the block output to adjust
 *	the html.
 *
 *	@note		This is being used instead of blocks.getSaveContent.extraProps (JS)
 *				because it does not currently work for render_callback blocks ???
 *				This also makes it safer when removed to prevent block errors.
 *
 *	@since 		v2.2.3
 *
*/
function bne_render_block_visibility( $block_content, $block ) {
	//bne_printr($block);
	
	// Set visibility classes
	$visibility_classes = array();
	if( isset( $block['attrs']['hideOnMobile'] ) && $block['attrs']['hideOnMobile'] == true ) {
		$visibility_classes[] = 'hide_on_mobile';
	} 
	if( isset( $block['attrs']['hideOnTablet'] ) && $block['attrs']['hideOnTablet'] == true ) {
		$visibility_classes[] = 'hide_on_tablet';
	} 
	if( isset( $block['attrs']['hideOnStandard'] ) && $block['attrs']['hideOnStandard'] == true ) {
		$visibility_classes[] = 'hide_on_standard';
	} 
	
	if( ! empty( $visibility_classes ) ) {

		// Replace the first occurance of class=" without classes.
		// We need the classes to be added directly to the blocks. Wrapping classes can sometimes block full width content.
		$needle = 'class="';

		// Find the first occurance
		$pos = strpos( $block_content, $needle );

		if ( $pos !== false ) {
			// Our classes
			$replacement = 'class="'.implode( ' ', $visibility_classes).' ';
			// Replace it
			$new_block = substr_replace( $block_content, $replacement, $pos, strlen( $needle ) );
		} else {
			// Fallback to wrapping classes when block has no exsisting classes
			$new_block = '<div class="'.implode( ' ', $visibility_classes).'">'.$block_content.'</div>';
		}
		
		return $new_block;
	
	} else {
		
		return $block_content;
	
	}

}
add_filter( 'render_block', 'bne_render_block_visibility', 10, 2 );




/*
 *	Register Dynamic BNE Block Types
 *	Some blocks a PHP component for rendering
 *	within the editor and frontend.
 *
 *	@since 		v2.1.1
 *	@updated 	v2.2.4
 *
*/
function bne_block_register_dynamic_block_types() {

	// Card
	// register_block_type( 'bne/card',
	// 	array(
	// 		'attributes'=>	array(
	// 			'blockVersion'		=>	array( 'type' => 'string', 'default' => '' ),
	// 			'className'			=>	array( 'type' => 'string', 'default' => '' ),
	// 			'mediaURL'			=>	array( 'type' => 'string', 'default' => '' ),
	// 			'mediaAlt'			=>	array( 'type' => 'string', 'default' =>	'' ),
	// 			'mediaAltCustom' 	=>	array( 'type' => 'string', 'default' =>	'' ),
	// 			'mediaPosition'		=>	array( 'type' => 'string', 'default' => 'top' ),
	// 			'mediaLink' 		=>	array( 'type' => 'string', 'default' => '' )
	// 		),
	// 		'render_callback' => 'bne_block_render_card',
	// 	)
	// );
	
	// Google Map
	register_block_type( 'bne/map',
		array(
			'attributes'=>	array(
				'className'			=>	array( 'type' => 'string', 'default' => '' ),
				'align'				=>	array( 'type' => 'string' ),
				'APIKey'			=>	array( 'type' => 'string', 'default' =>	get_option( 'bne_block_map_apikey' ) ),
				'address' 			=>	array( 'type' => 'string', 'default' =>	'Los Angeles, CA, USA' ),
				'mapType'			=>	array( 'type' => 'string', 'default' => 'roadmap'),
				'zoom' 				=>	array( 'type' => 'integer', 'default' => 15 ),
				'height' 			=>	array( 'type' => 'integer', 'default' => 250 ),
				'interactiveMap' 	=>	array( 'type' => 'boolean', 'default' => true ),
				'grayscale' 		=>	array( 'type' => 'boolean', 'default' => false ),
				'showAddress' 		=>	array( 'type' => 'boolean', 'default' => false ),
				'showDirections'	=>	array( 'type' => 'boolean', 'default' => false )
			),
			'render_callback' => 'bne_block_render_google_map',
		)
	);
	
	// Blog Post
	register_block_type( 'bne/posts',
		array(
			'attributes'=>	array(
				'className'			=>	array( 'type' => 'string', 'default' => '' ),
				'layout'			=>	array( 'type' => 'string', 'default' =>	'grid' ),
				'postStyle'			=>	array( 'type' => 'string', 'default' =>	'default' ),
				'cardShadow'		=>	array( 'type' => 'string', 'default' => 'shadow-none' ),
				'columns' 			=>	array( 'type' => 'number', 'default' =>	3 ),
				'categorySelected' 	=>	array( 'type' => 'string', 'default' => '' ),
				'limit'				=>	array( 'type' => 'number', 'default' => 6 ),
				'thumbnail' 		=>	array( 'type' => 'string', 'default' => 'full' ),
				'thumbnailStyle' 	=>	array( 'type' => 'string', 'default' => 'flat' ),
				'showExcerpt'		=>	array( 'type' => 'boolean', 'default' => true ),
				'showCategory'		=>	array( 'type' => 'boolean', 'default' => true ),
				'showDate'			=>	array( 'type' => 'boolean', 'default' => true ),
				'showAuthor'		=>	array( 'type' => 'boolean', 'default' => true ),
				'showComments'		=>	array( 'type' => 'boolean', 'default' => true ),
				'showReadmore'		=>	array( 'type' => 'boolean', 'default' => false ),
				'readmoreText'		=>	array( 'type' => 'string', 'default' => get_theme_mod('post_read_more_label', 'Read More' ) )
			),
			'render_callback' => 'bne_block_render_posts',
		)
	);

	// FA Icon
	register_block_type( 'bne/fa-icon',
		array(
			'attributes'=>	array(
				'className'		=>	array( 'type' => 'string', 'default' => '' ),
				'align'			=>	array( 'type' => 'string', 'default' =>	'' ),
				'icon'			=>	array( 'type' => 'string', 'default' =>	'address-book' ),
				'style'			=>	array( 'type' => 'string', 'default' =>	'fas' ),
				'color'			=>	array( 'type' => 'string', 'default' => 'inherit' ),
				'size' 			=>	array( 'type' => 'number', 'default' =>	2 ),
				'bgColor'			=>	array( 'type' => 'string' ),
				'bgBorderRadius'	=>	array( 'type' => 'number', 'default' => 10 ),
				'bgShadow'			=>	array( 'type' => 'string', 'default' => 'shadow-none' ),
				'linkURL'		=>	array( 'type' => 'string', 'default' => '' ),
				'linkTab'		=>	array( 'type' => 'boolean', 'default' => 0 ),
				'linkLightbox'	=>	array( 'type' => 'boolean', 'default' => 0 )
			),
			'render_callback' => 'bne_block_render_fa_icon',
		)
	);
	
	// Google Fonts
	register_block_type( 'bne/google-fonts',
		array(
			'attributes'=>	array(
				'className'			=>	array( 'type' => 'string', 'default' => '' ),
				'align'				=>	array( 'type' => 'string', 'default' => '' ),
				'blockType'			=>	array( 'type' => 'string', 'default' => 'p' ),
				'content'			=>	array( 'type' => 'string', 'default' => '' ),
				'fontID'			=>	array( 'type' => 'string', 'default' => '' ),
				'variant'			=>	array( 'type' => 'string', 'default' => '' ),
				'fontSize'			=>	array( 'type' => 'number', 'default' => '' ),
				'lineHeight'		=>	array( 'type' => 'number', 'default' => '' ),
				'color'				=>	array( 'type' => 'string', 'default' => '' )
			),
			'render_callback'	=>	'bne_block_render_google_fonts',
		)
	);


	// Modal
	register_block_type( 'bne/modal',
		array(
			'attributes'=>	array(
				'className'			=>	array( 'type' => 'string', 'default' => '' ),
				'modalID'			=>	array( 'type' => 'string', 'default' => '' ),
				'modalSize'			=>	array( 'type' => 'string', 'default' => 'md' ),
				'modalHeading'		=>	array( 'type' => 'string', 'default' => '' ),
			),
			'render_callback'	=>	'bne_block_render_modal',
		)
	);



}
add_action( 'init', 'bne_block_register_dynamic_block_types' );



/*
 *	BNE Dynamic Block Render: Card
 *
 *	Block Slug: bne/card
 *
 *	@since 		TODO
 *
*/
// function bne_block_render_card( $atts, $content ) {
// 	//bne_printr($atts);
// 	$output = '';
// 	bne_printr( parse_blocks($content) );
// 	
// 	if( $atts['blockVersion'] != 2 ) {
// 		$old_content = parse_blocks($content);
// 		$content = '';
// 		//bne_printr( $content );
// 	}
// 
// 	// Block classes	
// 	$block_classes = 'wp-block-bne-card bne-card card mb-3';
// 	$block_classes .= isset( $atts['className'] ) ? ' '.$atts['className'] : '';
// 
// 
// 
// 
// 	$output .= '<div class="'.$block_classes.'">';
// 	
// 		if( $atts['mediaPosition'] == 'top' ) {
// 			$output .= '<div class="card-img-top">';
// 				if( $atts['mediaLink'] ) {
// 					$output .= '<a href="'.esc_url( $atts['mediaLink'] ).'" class="card-link">';
// 				}
// 				
// 				$output .= wp_get_attachment_image(
// 					$atts['mediaID'],
// 					'full',
// 					'',
// 					array(
// 						'class' => 'card-image',
// 						'alt' => $atts['mediaAltCustom'] ? $atts['mediaAltCustom'] : esc_attr( $atts['mediaAlt'] )
// 					)
// 				);
// 	
// 				if( $atts['mediaLink'] ) {
// 					$output .= '</a>';
// 				}
// 			$output .= '</div>';
// 		}
// 	
// 		
// 		$output .= '<div class="card-body">';
// 			$output .= $content;
// 		$output .= '</div>';
// 	
// 	$output .= '</div>';
// 	
// 	
// 	return $output;
// }





/*
 *	BNE Dynamic Block Render: Google Map
 *
 *	Block Slug: bne/map
 *
 *	@since 		v2.1.3
 *	@updated	v2.2.4
 *
*/
function bne_block_render_google_map( $atts, $content ) {
	//bne_printr($atts);
	
	// Update Stored API Key
	if( $atts['APIKey'] ) {
		$current_saved_key = get_option( 'bne_block_map_apikey' );
		if( $atts['APIKey'] != $current_saved_key ) {
			sanitize_text_field( update_option( 'bne_block_map_apikey', $atts['APIKey'] ) );
		}
	}
	
	// Block classes	
	$block_classes = 'wp-block-bne-map bne-map';
	$block_classes .= isset( $atts['className'] ) ? ' '.$atts['className'] : '';
	$block_classes .= isset( $atts['align'] ) ? ' align'.$atts['align'] : '';
	$block_classes .= $atts['interactiveMap'] ? ' map-interactive' : ' map-image';
	$block_classes .= $atts['grayscale'] ? ' map-grayscale' : '';
	
	
	
	// Output
	$output = '<div class="'.$block_classes.'">';
		
		// Interactive (iframe) Map
		if( true == $atts['interactiveMap'] ) {
			$url = add_query_arg( array(
				'key'	=>	$atts['APIKey'],
				'q'		=>	urlencode( $atts['address'] ),
				'zoom'	=>	$atts['zoom'],
				'maptype' => $atts['mapType']
			), 'https://www.google.com/maps/embed/v1/place' );
			$output .= '<iframe width="100%" height="'.$atts['height'].'" frameborder="0" style="border:0" src="'.esc_url($url).'" allowfullscreen></iframe>';
			
		// Image Map
		} else {
			$url = add_query_arg( array(
				'key'		=>	$atts['APIKey'],
				'zoom'		=>	$atts['zoom'],
				'scale'		=>	2,
				'size'		=>	'600x'.$atts['height'],
				'maptype'	=>	$atts['mapType'],
				'format'	=>	'jpeg',
				'center'	=> 	urlencode( $atts['address'] ),
				'markers'	=>	'size:mid|'.urlencode( $atts['address'] )
			), 'https://maps.googleapis.com/maps/api/staticmap' );
			$output .= '<img src="'.esc_url($url).'" style="width:100%;"/>';
		}
	
		$output .= '<div class="bne-map-footer mt-3 bne-editor-row row">';
			
			if( true == $atts['showAddress'] ) {
				$output .= '<div class="map-address bne-editor-col col-md ">'.$atts['address'].'</div>';
			}
			
			if( true == $atts['showDirections'] ) {
				$output .= '<div class="map-directions bne-editor-col-auto col-md-auto">';
					$output .= '<a href="https://maps.google.com/?daddr='.urlencode( $atts['address'] ).'" class="btn btn-sm btn-default" target="_blank"><i class="bne-fa fa fa-car"></i> '.esc_html__( 'Get Directions', 'bne' ).'</a>';
				$output .= '</div>';
			}
		
		$output .= '</div>';
	$output .= '</div>';
    
    return $output;
}




/*
 *	BNE Dynamic Block Render: Posts
 *
 *	Block Slug: bne/posts
 *
 *	@since 		v2.1.6
 *	@updated	v2.2.14
 *
*/
function bne_block_render_posts( $atts, $content ) {
	//bne_printr($content);
	
	// Setup Query
	$query_args = array(
		'post_type' 			=> 	'post',
		'posts_per_page'		=> 	$atts['limit'],
		'ignore_sticky_posts' 	=> 	true,
		'cat'					=> 	$atts['categorySelected'],
	);
			
	// Block classes	
	$block_classes = 'wp-block-bne-posts bne-posts';
	$block_classes .= isset( $atts['className'] ) ? ' '.$atts['className'] : '';
	$block_classes .= ' has-layout-'.$atts['layout'];
	$block_classes .= ' has-post-style-'.$atts['postStyle'];
	$block_classes .= ($atts['categorySelected']) ? ' has-category-'.$atts['categorySelected'] : '';
	$block_classes .= ' has-thumbnail-'.$atts['thumbnail'];
	$block_classes .= ' has-thumbnail-style-'.$atts['thumbnailStyle'];
	$block_classes .= ($atts['showExcerpt'] ) ? ' has-meta-excerpt' : '';
	$block_classes .= ($atts['showCategory'] ) ? ' has-meta-category' : '';
	$block_classes .= ($atts['showDate'] ) ? ' has-meta-date' : '';
	$block_classes .= ($atts['showAuthor'] ) ? ' has-meta-author' : '';
	$block_classes .= ($atts['showComments'] ) ? ' has-meta-comments' : '';
	$block_classes .= ($atts['showReadmore'] ) ? ' has-meta-readmore' : '';

	// Columns
	if( $atts['columns'] >= 5 ) { $atts['columns'] = 5; $col = '15'; }
	else { $col = 12 / $atts['columns']; }
	if( $atts['layout'] == 'list' ) { $col = '12'; }
	
	// Thumbnail
	$thumb_crop = $atts['thumbnail'];
	$thumb_class = 'thumbnail-'.$atts['thumbnail'] .' alignleft';
	if( $thumb_crop == 'mini' ) { 
		$thumb_crop = 'thumbnail';
	} elseif( $thumb_crop == 'thumbnail' ) { 
		$thumb_class = 'thumbnail-small alignleft';
	}
	
	// Meta
	$meta = array();
	if( $atts['showDate'] ) { $meta[] = 'date'; }
	if( $atts['showAuthor'] ) { $meta[] = 'author'; }
	if( $atts['showCategory'] ) { $meta[] = 'category'; }
	if( $atts['showComments'] ) { $meta[] = 'comments'; }
	
	// Shadow
	if( ( 'default' == $atts['postStyle'] ) && ( 'portfolio' != $atts['layout'] ) ) {
		$atts['cardShadow'] = 'shadow-none';
	}
	
	// Portfolio Layout
	if( 'portfolio' == $atts['layout'] ) {
		$thumb_crop = 'hide';
		$block_classes .= ' text-light';
	}

	// Output
	ob_start();
	echo '<div class="'.$block_classes.'">';
		
		echo '<div class="bne-editor-row row">';
			
			// Start Query
			$posts = new WP_Query( $query_args );
			if( $posts->have_posts() ) {
				
				// The Loop
				while( $posts->have_posts() ) : $posts->the_post();
					
					echo '<div class="bne-editor-column bne-editor-col-'.$col.' col-post col-sm-'.$col.'">';
					
						echo '<article id="post-'.get_the_ID().'" class="'.$atts['postStyle'].' '.$atts['cardShadow'].'" style="--post-image: url('.get_the_post_thumbnail_url( null, $size = 'full').');">';
							
							echo bne_post_thumbnail( $thumb_crop, $thumb_class, 'card-img-top is-style-'.$atts['thumbnailStyle'] ); 
							
							echo '<div class="card-body">';
								echo '<header class="entry-header card-title">';
									echo '<h3 class="entry-title">';
										echo '<a href="'.get_the_permalink().'" title="'.the_title_attribute( array( 'echo' => false ) ).'">'.get_the_title().'</a>';
									echo '</h3>';
								echo '</header><!-- .entry-header -->';
								
								echo bne_post_meta('', $meta);
							
								echo '<div class="entry-content card-text">';
								
									if( true == $atts['showExcerpt'] ) {
										echo '<div class="entry-excerpt">';
											echo the_excerpt();
										echo '</div>';
									}
									
									if( true == $atts['showReadmore'] ) {
										if( !$atts['readmoreText'] ) {
											$atts['readmoreText'] = get_theme_mod('post_read_more_label', 'Read More' );
										}
										echo '<p class="entry-readmore"><a href="'.get_the_permalink().'" title="'.the_title_attribute( array( 'echo' => false ) ).'" class="btn btn-default btn-readmore">'.$atts['readmoreText'].'</a></p>';
									}
							
								echo '</div><!-- /.entry-content -->';
						
							echo '</div> <!-- /.card-body -->';
						echo '</article><!-- /#post -->';

					echo '</div>';
					
				endwhile;
			}
	
		echo '</div>';
	echo '</div>';
	
	// Reset Query
	wp_reset_postdata();
	
	return ob_get_clean();
}




/*
 *	BNE Dynamic Block Render: Font Awesome Icon
 *
 *	Block Slug: bne/fa-icon
 *
 *	@since 		v2.1.7
 *	@updated	v2.2.1
 *
*/
function bne_block_render_fa_icon( $atts, $content ) {
	//bne_printr($atts);
	
	// Block Classes
	$block_classes = 'wp-block-bne-fa-icon bne-vector-icon';
	$block_classes .= !empty( $atts['align'] ) ? ' align'.$atts['align'] : '';
	$block_classes .= isset( $atts['bgColor'] ) ? ' has-bg-color' : '';
	$block_classes .= ' fa-layers fa-fw fa-'.$atts['size'].'x';
	$block_classes .= !empty( $atts['className'] ) ? ' '.$atts['className'] : '';
	
	// Block Styles
	$block_styles = 'color:'.$atts['color'].';';
	
	// Background Styles
	if( !empty ( $atts['bgColor'] ) ) {
		$block_classes .= ' '.$atts['bgShadow'];
		$block_styles .= 'background:'.$atts['bgColor'].';border-radius:'.$atts['bgBorderRadius'].'%;';
	}
	
	// Output
	ob_start();
	echo '<div class="'.$block_classes.'" style="'.$block_styles.'">';
		echo '<div class="vector-icon-inner">';

			// Open Link
			if( $atts['linkURL'] ) {
				$target = ($atts['linkTab'] ? '_blank' : '_self' );
				$rel = ($atts['linkTab'] ? 'noreferrer noopener' : '' );
				$class = ($atts['linkLightbox'] ? 'bne-lightbox' : '' );
				echo '<a href="'.esc_url( $atts['linkURL'] ).'" target="'.$target.'" rel="'.$rel.'" class="'.$class.'">';
			}

			// Icon
			echo '<i class="bne-fa '.$atts['style'].' fa-'.$atts['icon'].'"></i>';

			// Close Link
			if( $atts['linkURL'] ) {
				echo '</a>';
			}

		echo '</div>';
	echo '</div>';
	return ob_get_clean();
}




/*
 *	BNE Dynamic Block Render: Google fonts
 *
 *	Block Slug: bne/google-fonts
 *
 *	@since 		v2.2.3
 *
*/
function bne_block_render_google_fonts( $atts, $content ) {
	$output = '';
	$styles = '';
	//bne_printr($atts);

	// Base Class
	$classes = 'wp-block-bne-google-fonts';
	
	// Get Block Atts
	$classes .= isset( $atts['className'] ) ? ' '.sanitize_text_field( $atts['className'] ) : '';
	$align = isset( $atts['align'] ) ? esc_attr( $atts['align'] ) : '';
	$blockType = isset( $atts['blockType'] ) ? esc_attr( $atts['blockType'] ) : 'p';
	$content = isset( $atts['content'] ) ? wp_kses_post( $atts['content'] ) : '';
	$fontID = isset( $atts['fontID'] ) ? sanitize_text_field( $atts['fontID'] ) : '';
	$weight = isset( $atts['variant'] ) ? sanitize_text_field( $atts['variant'] ) : '';
	$fontSize = isset( $atts['fontSize'] ) ? intval( $atts['fontSize'] ) : '';
	$lineHeight = isset( $atts['lineHeight'] ) ? intval( $atts['lineHeight'] ) : '';
	$color = isset( $atts['color'] ) ? sanitize_text_field( $atts['color'] ) : '';
	
	// Set Styles
	if( $fontID ) {
		$classes .= ' has-font-'.str_replace( ' ', '-', strtolower($fontID) );
		$styles .= 'font-family:'.str_replace( '+', ' ', $fontID ).';';
	}
	if( $weight ) {
		if( $weight != 'regular' ) {
			$styles .= 'font-weight:'.$weight.';';
		}
	}
	if( $fontSize ) {
		$styles .= 'font-size:'.$fontSize.'rem;';
	}
	if( $lineHeight ) {
		$styles .= 'line-height:'.$lineHeight.';';
	}
	if( $color ) {
		$styles .= 'color:'.$color.';';
	}
	if( $align ) {
		$styles .= 'text-align:'.$align.';';
	}
	
	// Enqueue font stylesheet
	if( $fontID ) {
		wp_enqueue_style( 
			'bne-blocks-google-fonts-'.str_replace( ' ', '-', strtolower($fontID) ).'-'.$weight, 
			add_query_arg( array(
				'family' => str_replace( ' ', '+', $fontID ).':'.$weight,
				'display' => 'swap'
			), '//fonts.googleapis.com/css' ), 
			array(), 
			BNE_FRAMEWORK_VERSION 
		);
	}	 	

	// Output
	$output = '<'.$blockType.' class="'.$classes.'" style="'.$styles.'">'.( $content ? $content : 'White a heading...' ).'</'.$blockType.'>';
	
	return $output;
}





/*
 *	BNE Dynamic Block Render: Modal
 *
 *	Block Slug: bne/modal
 *
 *	@since 		v2.2.4
 *
*/
function bne_block_render_modal( $atts, $content ) {
	//bne_printr($atts);
	$output = '';
	

	// Base Class
	$classes = 'wp-block-bne-modal modal fade';
	
	// Get Block Atts
	$classes .= isset( $atts['className'] ) ? ' '.sanitize_text_field( $atts['className'] ) : '';
	$modalID = isset( $atts['modalID'] ) ? sanitize_text_field( $atts['modalID'] ) : '';
	$size = isset( $atts['size'] ) ? sanitize_text_field( $atts['size'] ) : 'md';
	$heading = isset( $atts['heading'] ) ? sanitize_text_field( $atts['heading'] ) : '';
	$autoOpen = isset( $atts['autoOpen'] ) ? intval( $atts['autoOpen'] ) : '';


	// Inline JS to trigger the modal on click
	wp_add_inline_script( 'bootstrap', '
		jQuery(document).ready(function($) {
			$(".modal-trigger-'.$modalID.'").on( "click", function(e) {
				e.preventDefault();
				$("#modal-id-'.$modalID.'").modal();
			});
			'.( $autoOpen ? '$("#modal-id-'.$modalID.'").modal("show");' : '' ).'
		});
	', 'after' );
	
	// Modal Content
	$modal = '<div id="modal-id-'.$modalID.'" class="'.$classes.'" tabindex="-1" role="dialog" aria-labelledby="modal-title-'.$modalID.'">';
		$modal .= '<div class="modal-dialog modal-'.$size.'" role="document">';
			$modal .= '<div class="modal-content">';
				$modal .= '<div class="modal-header">';
					$modal .= '<h5 class="modal-title" id="modal-title-'.$modalID.'">'.$heading.'</h5>';
					$modal .= '<button type="button" class="close" data-dismiss="modal" aria-label="Close"><i class="bne-fa fas fa-times"></i></button>';
				$modal .= '</div>';
				$modal .= '<div class="modal-body">'.$content.'</div>';
			$modal .= '</div>';
		$modal .= '</div>';
	$modal .= '</div>';
	
	// Throw modals within the footer and not within the page content
	add_action( 'wp_footer', function() use ( $modal ) {
	    echo do_shortcode($modal);
	});
}