/*
 * 	BNE Block: Google Fonts
 *
 * 	@author		Kerry Kline
 * 	@copyright	Copyright (c) Kerry Kline
 * 	@link		http://www.bnecreative.com
 *
 * 	@updated:	April 15, 2020
 *
*/


( function( blocks, components, i18n, element, editor ) {
	var registerBlockType = wp.blocks.registerBlockType;
	var el = wp.element.createElement;
	var __ = wp.i18n.__;
	var InspectorControls = wp.blockEditor.InspectorControls;
	var BlockControls = wp.blockEditor.BlockControls;
	var AlignmentToolbar = wp.blockEditor.AlignmentToolbar;
	var ColorPalette = wp.blockEditor.ColorPalette;
	var RichText = wp.blockEditor.RichText;
	var PanelBody = wp.components.PanelBody;
	var SelectControl = wp.components.SelectControl;
	var RangeControl = wp.components.RangeControl;



	// Get Google Fonts List
	var fontsJson = bne_blocks_google_fonts_vars.fontsJson;
	//console.log( fontsJson.items );

	
	// Block Icon
	var blockIcon = el( 'svg', { className: 'components-bne-svg', width: '24', height: '24', 'viewbox': '0 0 24 24', 'baseProfile': 'tiny', xmlns: 'http://www.w3.org/2000/svg' }, el( 'path', { fill: 'none', d: "M0 0h24v24H0V0z" } ), el( 'path', { d: "M9.93 13.5h4.14L12 7.98zM20 2H4c-1.1 0-2 .9-2 2v16c0 1.1.9 2 2 2h16c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2zm-4.05 16.5l-1.14-3H9.17l-1.12 3H5.96l5.11-13h1.86l5.11 13h-2.09z" } ) );
	

	/*
	 * Get font families for use in <select>.
	 *
	 * @returns {Object}  value/label pair.
	*/
	function getFontsForSelect(fontsJson) {		
		const googleFonts = fontsJson.items.map( ( font ) => {
			const label = font.family;
			const value = label;
			//const value = label.replace( /\s+/g, '+' );

			return {
				value: value,
				label: label,
			};
		} );
		
		googleFonts.unshift( { label: '- Select Font -', value: '' } );

		return googleFonts;
	}



	/*
	 * All the font weights as options to be used in a <select> element
	 *
	 * @param   {string} fontFamily font-family name.
	 * @returns {Object}  The font object.
	*/
	function getFontObject( fontFamily ) {
		if ( ! fontFamily ) {
			return;
		}

		// iterate over each element in the array
		for ( let i = 0; i < fontsJson.items.length; i++ ) {
			// look for the entry with a matching `code` value
			if ( fontsJson.items[ i ].family === fontFamily ) {
				return fontsJson.items[ i ];
			}
		}
	}


	/*
	 * Check if a font weight is italic.
	 *
	 * @param   {string} value A font weight.
	 * @returns {boolean}  false is value is italic.
	*/
	function isItalic( value ) {
		if ( value.includes( '0i' ) || value === 'italic' ) {
			return false;
		}
		return true;
	}


	/*
	 * Get font variants for use in <select>.
	 *
	 * @param   {Object} fontObject The font object.
	 * @returns {Object} value/label pair.
	*/
	function getVariantsForSelect( fontObject ) {
		if ( ! fontObject ) {
			return;
		}
		
		const variants = getFontObject(fontObject).variants.map( ( v ) => {
			return {
				value: v,
				label: v,
			};
		} );
		
		// Remove non numerical variants
		return variants.filter(function(hero) {
			if ( hero.value.includes( '0i' ) || hero.value === 'italic' ) {
				return false;
			} else {
				return true;
			}
		});
		
	}



	/*
	 * Add Google Font link to head in block editor.
	 *
	 * @param {string} fontFamily font-family name.
	 * @param {Object} fontObject The font object.
	*/
	function addGoogleFontToHead( fontFamily, variants ) {
		if ( ! fontFamily || ! variants ) {
			return;
		}

		const head = document.head;
		const link = document.createElement( 'link' );
		
		link.type = 'text/css';
		link.rel = 'stylesheet';
		link.href = 'https://fonts.googleapis.com/css?family=' + fontFamily.replace( /\s+/g, '+' ) + ':' + variants;

		head.appendChild( link );
	}
	
	
	

	// ========================== //
	// Register Block
	// ========================== //
	registerBlockType( 'bne/google-fonts', {
		title: __( 'Google Fonts', 'bne' ),
		description: __( 'A custom Google Typography block for headings and general text.', 'bne' ),
		icon: blockIcon,
		category: 'bne-blocks',
		keywords: [
			__( 'Fonts', 'olympus-google-fonts' ),
			__( 'Heading', 'olympus-google-fonts' ),
		],
		attributes: {
			
			// BlockControls
			align: {
				type: 'string',
				default: 'left'
			},
			
			// Font Settings
			blockType: {
				type: 'string',
				default: 'p'
			},
			content: {
				type: 'string',
				default: ''
			},
			fontID: {
				type: 'string',
				default: ''
			},
			variant: {
				type: 'string',
				default: 'regular'
			},
			fontSize: {
				type: 'integer',
				default: ''
			},
			lineHeight: {
				type: 'integer',
				default: ''
			},
			color: {
				type: 'string',
				default: 'inherit'
			}
		},

		transforms: {
		    from: [
		        {
		            type: 'block',
		            blocks: [ 'core/heading' ],
		            transform: function ( attributes ) {
		                return wp.blocks.createBlock( 'bne/google-fonts', {
		                    content: attributes.content,
		                    align: attributes.align
		                } );
		            },
		        },
		    ],
		    to: [
		        {
		            type: 'block',
		            blocks: [ 'core/heading' ],
		            transform: function ( attributes ) {
		                return wp.blocks.createBlock( 'core/heading', {
		                    content: attributes.content,
		                    align: attributes.align
		                } );
		            },
		        },
		    ]
		},	


		
		// ========================== //
		// Admin Edit
		// ========================== //
		edit: function( props ) {
			var attributes = props.attributes;
			//console.log(attributes);
			
			// Enqueue the font face to the editor
			if( attributes.fontID ) {
				addGoogleFontToHead( attributes.fontID.replace( /\s+/g, '+' ), attributes.variant );
			}


 			// Return Output
			// ========================== //
 			return [

	  			// Block Controls
  				el( BlockControls, {},
					el( AlignmentToolbar, {
						value: !attributes.align ? 'left' : attributes.align,
						onChange: function( newAlign ) {
							props.setAttributes( { align: newAlign } );
						},
					} )
				),

				// Panel Controls
				el( InspectorControls, {},

 					// Font Settings Panel
 					el( PanelBody, {
							title: __( 'Font Settings', 'bne' ),
							initialOpen: true,
						},
						
						// Block Div
						el( SelectControl, {
								value: attributes.blockType,
								label: __( 'Block Level', 'bne' ),
								onChange: function( newBlockType ) {
									props.setAttributes( { blockType: newBlockType } );
								},
								options: [
									{ value: 'p', label: __( 'Paragraph', 'bne' ) },
									{ value: 'h1', label: __( 'H1' ) },
									{ value: 'h2', label: __( 'H2' ) },
									{ value: 'h3', label: __( 'H3' ) },
									{ value: 'h4', label: __( 'H4' ) },
									{ value: 'h5', label: __( 'H5' ) },
									{ value: 'h6', label: __( 'H6' ) },
									{ value: 'span', label: __( 'Span' ) },
								]
							} 
						),
						
						// Font Family
						el( SelectControl, {
								value: attributes.fontID,
								label: __( 'Font', 'bne' ),
								onChange: function( newFontID ) {
									props.setAttributes( { fontID: newFontID } );
									props.setAttributes( { variant: 'regular' } );
								},
								options: getFontsForSelect(fontsJson)
							} 
						),
						
						// Font Weight
						el( SelectControl, {
								value: attributes.variant,
								label: __( 'Font Variant', 'bne' ),
								onChange: function( newVariant ) {
									props.setAttributes( { variant: newVariant } );
								},
								options: getVariantsForSelect( attributes.fontID )
							} 
						),

						// Font Size
						el( RangeControl, {
							className: "components-font-size-picker__custom-input",
							label: __( 'Font Size (rem)', 'bne' ),
							value: attributes.fontSize,
							initialPosition: 1,
							onChange: function( newFontSize ) {
									props.setAttributes( { fontSize: newFontSize } );
							},
							min: 1,
							max: 10,
							step: '0.1',
							allowReset: true,
						} ),
						
						// Line Height
						el( RangeControl, {
							className: "components-font-size-picker__custom-input",
							label: __( 'Line Height', 'bne' ),
							value: attributes.lineHeight,
							initialPosition: 1,
							onChange: function( newLineHeight ) {
									props.setAttributes( { lineHeight: newLineHeight } );
							},
							min: 1,
							max: 3,
							step: '0.1',
							allowReset: true,
						} ),
						
						// Text Color
						el('p', {}, __( 'Text Color', 'bne' ) ),
						el( ColorPalette, {
								value: attributes.color,
								onChange: function( newColor ) {
									props.setAttributes( { color: newColor } );
								},
							},
						),

					), // end PanelBody

				), // End InspectorControls



				// Editor
				// ========================== //
				el( RichText, {
					tagName: attributes.blockType ? attributes.blockType : 'p',
					className: props.className,
					placeholder: i18n.__( 'Write a heading...', 'bne' ),
					formattingControls: [ 'italic', 'link' ],
					keepPlaceholderOnFocus: true,
					value: attributes.content,
					onChange: function( newContent ) {
						props.setAttributes( { content: newContent } );
					},
					style: {
						fontSize: attributes.fontSize ? attributes.fontSize + 'rem' : undefined,
						textAlign: attributes.align,
						fontFamily: attributes.fontID.replace( /\+/g, ' ' ),
						fontWeight: attributes.variant,
						lineHeight: attributes.lineHeight,
						color: attributes.color,
					}
				} )

			]; // END return
		}, // END edit
	
		
		// ========================== //
		// Save
		// ========================== //
		save: function() {
       	 	// Rendering in PHP
	   	 	return null;
		} // End save

	} ); // End registerBlockType


} )(
	window.wp.blocks,
	window.wp.components,
	window.wp.i18n,
	window.wp.element,
	window.wp.editor,
);