<?php

// Exit if accessed directly
if ( !defined('ABSPATH')) exit;


/*
 *	BNE_Widgets_Post_List Class
 *	Extends the WP_Widget class with ours, BNE_Widgets_Post_List,
 *	to create a new available widget.
 *
 *  @since		v1.0
 *	@updated	v1.2
 *
*/
class BNE_Widgets_Post_List extends WP_Widget {
 
	public function __construct() {
	
		parent::__construct(
			'bne_widgets_post_list',
			esc_html__( 'BNE Post List', 'bne-widgets' ),
			array(
				'classname'   => 'bne_widgets_post_list',
				'description' => esc_html__( 'Display recent blog post.', 'bne-widgets' )
			)
		);

		add_action( 'widgets_init', function() {
            register_widget( 'BNE_Widgets_Post_List' );
        });

	} // End Construct
	
	
	
	/*  
	 * 	Front-end display of widget.
	 *
	 * 	@see WP_Widget::widget()
	 *
	 * 	@param array $args     Widget arguments.
	 * 	@param array $instance Saved values from database.
	*/
	public function widget( $args, $instance ) {
		
		// Saved Options
		$title = ( !empty( $instance['title'] ) ) ? apply_filters( 'widget_title', $instance['title'] ) : '';
		$category = ( !empty( $instance['category'] ) ) ? esc_attr( $instance['category'] ) : '';
		$more_text = ( !empty( $instance['more_text'] ) ) ? esc_attr( $instance['more_text'] ) : 'Read More';
		$meta = ( !empty( $instance['meta'] ) ) ? esc_attr( $instance['meta'] ) : 'true';
		$excerpt = ( !empty( $instance['excerpt'] ) ) ? esc_attr( $instance['excerpt'] ) : 'true';
		$thumb = ( !empty( $instance['thumb'] ) ) ? esc_attr( $instance['thumb'] ) : 'thumbnail';
		$thumb_class = '';

		// Setup Query
		$query_args = array(
			'post_type' 			=> 	'post',
			'posts_per_page'		=> 	$instance['limit'],
			'ignore_sticky_posts' 	=> 	true,
			'category_name'			=> 	$instance['category'],
		);

		// Empty string
		$output = '';

		echo $args['before_widget'];
		if( $title ) {
			echo $args['before_title'] . $title . $args['after_title'];
		}

		// Start Query
		$bne_post_list = new WP_Query( $query_args );
		if( $bne_post_list->have_posts() ) {
	
			$output .= '<div class="post-list-wrapper clearfix">';
	
				// The Loop
				while( $bne_post_list->have_posts() ) : $bne_post_list->the_post();
	
					$output .= '<article id="post-'.get_the_ID().'" class="mb-2 clearfix">';

						// Thumbnail
						$image_style = get_theme_mod( 'post_archive_thumbnail_style', 'flat' );
						if( $thumb == 'mini' ) { 
							$thumb = 'thumbnail';
							$thumb_class = 'thumbnail-mini alignleft mb-1';
						}
						$output .= bne_post_thumbnail( $thumb, $thumb_class, 'is-style-'.$image_style );

						// Post Head
						$output .= '<header class="entry-header">';
	
							// Post Title
							$output .= '<h4 class="entry-title">';
								$output .= '<a href="'.get_the_permalink().'" title="'.the_title_attribute( array( 'echo' => false ) ).'">'.get_the_title().'</a>';
							$output .= '</h4>';
	
							// Post Meta
							if( 'true' == $meta ) {
								$output .= bne_post_meta();
							} elseif ( 'date' == $meta ) {
								$output .= bne_post_meta( '', 'date' );
							}
	
						$output .= '</header><!-- .entry-header -->';
	
						// Post Content
						if( $excerpt == 'true' ) { 
							$output .= '<div class="entry-content">';
								$output .= '<p class="excerpt">'.get_the_excerpt().'</p>';
		
								// Post Link
								$output .= '<p class="read-more">';
									$output .= '<a href="'.get_the_permalink().'" title="'.the_title_attribute( array( 'echo' => 0 ) ).'" class="btn btn-default">'.esc_html( $more_text ).'</a>';
								$output .= '</p>';
							$output .= '</div><!-- .entry-content -->';
						}
								
							
					$output .= '</article><!-- #post-'.get_the_ID().'" (end) -->';
	
				endwhile;
	
			$output .= '</div><!-- .post-list-wrapper (end) -->';
	
		// No Post to display...
		} else {
			$output .= '<div class="alert alert-warning">'.esc_html__( 'There are no post to display.', 'bne' ).'</div>';
		}
		
		// Reset Query
		wp_reset_postdata();
	

		// Return data
		echo $output;

		echo $args['after_widget'];
	
	} // End widget
 
  
	/*
	 *  Sanitize widget form values as they are saved.
	 *
	 *  @see WP_Widget::update()
	 *
	 *  @param array $new_instance Values just sent to be saved.
	 *  @param array $old_instance Previously saved values from database.
	 *
	 *  @return array Updated safe values to be saved.
	*/
	public function update( $new_instance, $old_instance ) {        
	
		$instance = $old_instance;
	
		$instance['title'] = strip_tags( $new_instance['title'] );
		$instance['limit'] = strip_tags( $new_instance['limit'] );
		$instance['category'] = strip_tags( $new_instance['category'] );
		$instance['more_text'] = strip_tags( $new_instance['more_text'] );
		$instance['meta'] = strip_tags( $new_instance['meta'] );
		$instance['thumb'] = strip_tags( $new_instance['thumb'] );
		$instance['excerpt'] = strip_tags( $new_instance['excerpt'] );
	
		return $instance;
	
	} // End Update
	
	/*
	 *  Back-end widget form.
	 *
	 *  @see WP_Widget::form()
	 *
	 *  @param array $instance Previously saved values from database.
	*/
	public function form( $instance ) {    
	
		if( $instance ) {
			$title = sanitize_text_field( $instance['title'] );
			$limit = esc_attr($instance['limit']);
			$category = esc_attr($instance['category']);
			$more_text = sanitize_text_field( $instance['more_text'] );
			$meta = esc_attr( $instance['meta'] );
			$thumb = esc_attr( $instance['thumb'] );
			$excerpt = esc_attr( $instance['excerpt'] );
		} else {
			$title = 'Recent Post';
			$limit = '5';
			$category = '';
			$more_text = get_theme_mod( 'post_read_more_label', 'Read More' );
			$meta = 'true';
			$thumb = 'thumbnail';
			$excerpt = 'true';
		}
		
		?>
	
		<!-- Widget Title -->
		<p>
			<label for="<?php echo $this->get_field_id('title'); ?>"><?php _e('Widget Title', 'bne-widgets'); ?></label> 
			<input class="widefat" id="<?php echo $this->get_field_id('title'); ?>" name="<?php echo $this->get_field_name('title'); ?>" type="text" value="<?php echo $title; ?>" />
		</p>

		<!-- Limit -->
		<p>
			<label for="<?php echo $this->get_field_id('limit'); ?>"><?php _e('Post Limit', 'bne-widgets'); ?>:</label>
			<span style="display:block;padding:2px 0" class="description"><?php echo _e( 'Display this many post.', 'bne-widgets'); ?></span>
			<input class="widefat" id="<?php echo $this->get_field_id('limit'); ?>" name="<?php echo $this->get_field_name('limit'); ?>" type="number" value="<?php echo $limit; ?>" />
		</p>

		<!-- Category -->
		<p>
			<label for="<?php echo $this->get_field_id('category'); ?>"><?php _e('Post Category', 'bne-widgets'); ?>:</label>
			<span style="display:block;padding:2px 0" class="description"><?php echo _e( 'Use this category to display post from.', 'bne-widgets'); ?></span>
			<?php
			$category_args = array(
				'name'				=>	$this->get_field_name('category'),
				'show_option_none'	=>	__( 'Select category', 'bne-widgets' ),
				'option_none_value'	=> 	'',
				'value_field'		=>	'name',
				'show_count'       	=>	1,
				'orderby'          	=>	'name',
				'echo'             	=>	0,
				'selected'         	=>	$category,
				'class'            	=>	'widefat'
			);
			echo wp_dropdown_categories($category_args);
			?>
		</p>

		<!-- Read More Label -->
		<p>
			<label for="<?php echo $this->get_field_id('more_text'); ?>"><?php _e('Read More Text Label', 'bne-widgets'); ?>:</label>
			<span style="display:block;padding:2px 0" class="description"><?php echo _e( 'The label used for the "read more" button. Defaults to what is set in Theme Options.', 'bne-widgets'); ?></span>
			<input class="widefat" id="<?php echo $this->get_field_id('more_text'); ?>" name="<?php echo $this->get_field_name('more_text'); ?>" type="text" value="<?php echo $more_text; ?>" />
		</p>

		<!-- Meta -->
		<p>
			<label for="<?php echo $this->get_field_id('meta'); ?>"><?php _e('Post Meta', 'bne-widgets'); ?>:</label>
			<span style="display:block;padding:2px 0" class="description"><?php echo _e( 'Show or hide the post meta.', 'bne-widgets'); ?></span>
			<select name="<?php echo $this->get_field_name('meta'); ?>" id="<?php echo $this->get_field_id('meta'); ?>" class="widefat">
				<?php
					echo '<option value="true" id="true"', $meta == 'true' ? ' selected="selected"' : '', '>All Meta</option>';
					echo '<option value="date" id="date"', $meta == 'date' ? ' selected="selected"' : '', '>Date Only</option>';
					echo '<option value="false" id="false"', $meta == 'false' ? ' selected="selected"' : '', '>Hide</option>';
				?>
			</select>
		</p>

		<!-- Excerpt -->
		<p>
			<label for="<?php echo $this->get_field_id('excerpt'); ?>"><?php _e('Post Excerpt', 'bne-widgets'); ?>:</label>
			<span style="display:block;padding:2px 0" class="description"><?php echo _e( 'Show or hide the post excerpt.', 'bne-widgets'); ?></span>
			<select name="<?php echo $this->get_field_name('excerpt'); ?>" id="<?php echo $this->get_field_id('excerpt'); ?>" class="widefat">
				<?php
					echo '<option value="true" id="true"', $excerpt == 'true' ? ' selected="selected"' : '', '>Show</option>';
					echo '<option value="false" id="false"', $excerpt == 'false' ? ' selected="selected"' : '', '>Hide</option>';
				?>
			</select>
		</p>

		<!-- Thumbnail -->
		<p>
			<label for="<?php echo $this->get_field_id('thumb'); ?>"><?php _e('Thumbnail Size', 'bne-widgets'); ?>:</label>
			<span style="display:block;padding:2px 0" class="description"><?php echo _e( 'The post thumbnail size to display.', 'bne-widgets'); ?></span>
			<select name="<?php echo $this->get_field_name('thumb'); ?>" id="<?php echo $this->get_field_id('thumb'); ?>" class="widefat">
				<?php
					echo '<option value="mini" id="mini"', $thumb == 'mini' ? ' selected="selected"' : '', '>'.esc_html__('Mini', 'bne-widgets').'</option>';
					echo '<option value="thumbnail" id="thumbnail"', $thumb == 'thumbnail' ? ' selected="selected"' : '', '>'.esc_html__('Thumbnail', 'bne-widgets').'</option>';
					echo '<option value="medium" id="medium"', $thumb == 'medium' ? ' selected="selected"' : '', '>'.esc_html__('Medium', 'bne-widgets').'</option>';
					echo '<option value="large" id="large"', $thumb == 'large' ? ' selected="selected"' : '', '>'.esc_html__('Large', 'bne-widgets').'</option>';
					echo '<option value="full" id="full"', $thumb == 'full' ? ' selected="selected"' : '', '>'.esc_html__('Full', 'bne-widgets').'</option>';
					echo '<option value="hide" id="hide"', $thumb == 'hide' ? ' selected="selected"' : '', '>'.esc_html__('Hide', 'bne-widgets').'</option>';
				?>
			</select>
		</p>


    <?php 
    
    } // End Form
     
}
$BNE_Widgets_Post_List = new BNE_Widgets_Post_List();