<?php

// Exit if accessed directly
if ( !defined('ABSPATH')) exit;



/*
 *	BNE_Widgets_Map Class
 *	Extends the WP_Widget class with ours, BNE_Widgets_Map,
 *	to create a new available widget.
 *
 *  @since		v1.0
 *
*/
class BNE_Widgets_Map extends WP_Widget {
 
	public function __construct() {
	
		parent::__construct(
			'bne_widgets_map',
			esc_html__( 'BNE Google Map', 'bne-widgets' ),
			array(
				'classname'   => 'bne_widgets_map',
				'description' => esc_html__( 'Displays a Google map on your page. Requires a Google Map API Key.', 'bne-widgets' )
			)
		);

		add_action( 'widgets_init', function() {
            register_widget( 'bne_widgets_map' );
        });

	} // End Construct
	
	
	
	/*  
	 * 	Front-end display of widget.
	 *
	 * 	@see WP_Widget::widget()
	 *
	 * 	@param array $args     Widget arguments.
	 * 	@param array $instance Saved values from database.
	*/
	public function widget( $args, $instance ) {    
	
		// Saved Options
		$title = apply_filters( 'widget_title', $instance['title'] );
		$atts = array(
			'key'				=>	( !empty( $instance['key'] ) ) ? esc_attr( $instance['key'] ) : '',
			'address'			=> 	( !empty( $instance['address'] ) ) ? esc_attr( $instance['address'] ) : 'Los Angeles CA',
			'width'				=> 	( !empty( $instance['width'] ) ) ? esc_attr( $instance['width'] ) : '100%',
			'height'			=> 	( !empty( $instance['height'] ) ) ? esc_attr( $instance['height'] ) : '250px',
			'zoom'				=>	( !empty( $instance['zoom'] ) ) ? esc_attr( $instance['zoom'] ) : 15,
			'enablescrollwheel'	=>	( !empty( $instance['enablescrollwheel'] ) ) ? esc_attr( $instance['enablescrollwheel'] ) : 'true',
			'draggable'			=>	( !empty( $instance['draggable'] ) ) ? esc_attr( $instance['draggable'] ) : 'true',
			'disablecontrols'	=>	( !empty( $instance['disablecontrols'] ) ) ? esc_attr( $instance['disablecontrols'] ) : 'false',
			'icon'				=>	( !empty( $instance['icon'] ) ) ? esc_attr( $instance['icon'] ) : '',
			'static'			=>	( !empty( $instance['static'] ) ) ? esc_attr( $instance['static'] ) : 'false',
			'style'				=>	( !empty( $instance['style'] ) ) ? esc_attr( $instance['style'] ) : 'roadmap',
			'show_address'		=>	( !empty( $instance['show_address'] ) ) ? esc_attr( $instance['show_address'] ) : 'false',
			'show_directions'	=>	( !empty( $instance['show_directions'] ) ) ? esc_attr( $instance['show_directions'] ) : 'false',
		);
		

		if( !$atts['key'] || !$atts['address'] ) {
			return;
		}

		
		echo $args['before_widget'];
		if( $title ) {
			echo $args['before_title'] . $title . $args['after_title'];
		}
		
			
		// Static Map
		if( $atts['static'] == 'true' ) {
			$width = preg_replace( "/[^0-9,.]/", "", $atts['width'] );
			$height = preg_replace( "/[^0-9,.]/", "", $atts['height'] );
			$url = 'https://maps.googleapis.com/maps/api/staticmap?zoom='.esc_attr( $atts['zoom'] ).'&size='.$width.'x'.$height.'&scale=2&center='.urlencode( $atts['address'] ).'&format=jpg&markers='.urlencode( $atts['address'] ).'&maptype='.$atts['style'].'&key='.sanitize_text_field( $atts['key'] );
			
			echo '<img class="bne-static-map" width="'.$width.'" height="'.$height.'" alt="Map" src="'.$url.'">';
		
		
		// Embed Live Map
		} else {		
			
			// Enqueue Google API
			wp_enqueue_script( 'google-maps-api', '//maps.google.com/maps/api/js?Map&key='.sanitize_text_field( $atts['key'] ) );
		
			// Define Address Coordinates
			$address_hash = 'bne_sweetness_google_coordinates_'.md5( $atts['address'] );	
			
			// Retrieve cached coordinates
			$coordinates = get_transient( $address_hash );
			
			// Coordinates were not found in cache
			// then let's create one for this address (md5).
			if( $coordinates === false ) {
				$args = apply_filters( 'bne_map_query_args', array( 
					'address' 	=> 	urlencode( $atts['address'] ), 
					'sensor' 	=> 	'false',
					'key'		=> 	sanitize_text_field( $atts['key'] )
				) );
				$url = add_query_arg( $args, 'https://maps.googleapis.com/maps/api/geocode/json' );
				$response = wp_remote_get( $url );
				$data = wp_remote_retrieve_body( $response );
				
				if( is_wp_error( $data ) )
					return;
				
				if( $response['response']['code'] == 200 ) {
					$data = json_decode( $data, true );
					$coordinates = array(
						'lat' 	=> 	$data['results'][0]['geometry']['location']['lat'],
						'lng' 	=> 	$data['results'][0]['geometry']['location']['lng'],
					);
					// Cache the coordinates for 1 week
					set_transient( $address_hash, $coordinates, WEEK_IN_SECONDS );	
				}
			}
		
		
			// Draw the map if we have coordinates
			if( $atts['address'] && is_array( $coordinates ) ) {
				
				if( !is_admin() ) {
					wp_print_scripts( 'google-maps-api' );
				}
		
				// generate a unique ID for this map
				$map_id = uniqid( 'bne_map_' );
				
				// Map Script
				$scrollwheel = ( 'true' === strtolower( $atts['enablescrollwheel'] ) ) ? '1' : '0';
				$disableDefaultUI = ( 'true' === strtolower( $atts['disablecontrols'] ) ) ? '1' : '0';
				$draggable = ( 'true' === strtolower( $atts['draggable'] ) ) ? '1' : '0';
				$icon = ( $atts['icon'] ) ? $icon = 'icon: \''.esc_url( $atts['icon'] ).'\'' : '';
				wp_add_inline_script( 'bne_framework', '
					var map_'.$map_id.';
					function bne_map_'.$map_id.'(){
						var location = new google.maps.LatLng("'.$coordinates['lat'].'", "'.$coordinates['lng'].'");
						var map_options = {
							zoom: '.esc_attr( $atts['zoom'] ).',
							center: location,
							scrollwheel: '.$scrollwheel.',
							disableDefaultUI: '.$disableDefaultUI.',
							draggable: '.$draggable.',
							mapTypeId: \''.$atts['style'].'\'
						}
						map_'.$map_id.' = new google.maps.Map(document.getElementById("'.$map_id.'"), map_options);
						var marker = new google.maps.Marker({
							position: location,
							map: map_'.$map_id.',
							'.$icon.'	
						});
					}
					bne_map_'.$map_id.'();
				');
				
				ob_start();
				?>
				<div id="<?php echo esc_attr( $map_id ); ?>" class="bne-map" style="height: <?php echo esc_attr( $atts['height'] ); ?>; width: <?php echo esc_attr( $atts['width'] ); ?>"></div>
				<?php if( 'true' == $atts['show_address'] || 'true' == $atts['show_directions'] ) { ?>
					<div class="bne-map-footer clearfix">
						<?php if( 'true' == $atts['show_address'] ) { ?>
							<div class="map-address float-left"><?php echo $atts['address']; ?></div>
						<?php } ?>
						<?php if( 'true' == $atts['show_directions'] ) { ?>
							<div class="map-directions float-right"><a href="https://maps.google.com/?daddr=<?php echo $atts['address']; ?>" class="btn btn-sm btn-default" target="_blank"><i class="bne-fa fa fa-car"></i> <?php echo esc_html__( 'Get Directions', 'bne' ); ?></a></div>
						<?php } ?>
					</div>
				<?php } ?>
				<?php
				
				echo ob_get_clean();
			
			} else {
				echo esc_html__( 'This Map cannot be loaded because the maps API does not appear to be loaded.', 'bne' );
			}
		
		}
		
		echo $args['after_widget'];
	
	} // End widget
 
  
	/*
	 *  Sanitize widget form values as they are saved.
	 *
	 *  @see WP_Widget::update()
	 *
	 *  @param array $new_instance Values just sent to be saved.
	 *  @param array $old_instance Previously saved values from database.
	 *
	 *  @return array Updated safe values to be saved.
	*/
	public function update( $new_instance, $old_instance ) {        
		$instance = $old_instance;
	
		$instance['title'] = strip_tags( $new_instance['title'] );
		$instance['key'] = strip_tags( $new_instance['key'] );
		$instance['address'] = strip_tags( $new_instance['address'] );
		$instance['width'] = strip_tags( $new_instance['width'] );
		$instance['height'] = strip_tags( $new_instance['height'] );
		$instance['zoom'] = strip_tags( $new_instance['zoom'] );
		$instance['enablescrollwheel'] = strip_tags( $new_instance['enablescrollwheel'] );
		$instance['draggable'] = strip_tags( $new_instance['draggable'] );
		$instance['disablecontrols'] = strip_tags( $new_instance['disablecontrols'] );
		$instance['static'] = strip_tags( $new_instance['static'] );
		$instance['style'] = strip_tags( $new_instance['style'] );
		$instance['show_address'] = strip_tags( $new_instance['show_address'] );
		$instance['show_directions'] = strip_tags( $new_instance['show_directions'] );

		return $instance;
	
	} // End Update
	
	/*
	 *  Back-end widget form.
	 *
	 *  @see WP_Widget::form()
	 *
	 *  @param array $instance Previously saved values from database.
	*/
	public function form( $instance ) {    
	
		if( $instance ) {
			$title = sanitize_text_field( $instance['title'] );
			$key = esc_attr( $instance['key'] );
			$address = esc_attr( $instance['address'] );
			$width = esc_attr( $instance['width'] );
			$height = esc_attr( $instance['height'] );
			$zoom = esc_attr( $instance['zoom'] );
			$enablescrollwheel = esc_attr( $instance['enablescrollwheel'] );
			$draggable = esc_attr( $instance['draggable'] );
			$disablecontrols = esc_attr( $instance['disablecontrols'] );
			$static = esc_attr( $instance['static'] );
			$style = esc_attr( $instance['style'] );
			$show_address = esc_attr( $instance['show_address'] );
			$show_directions = esc_attr( $instance['show_directions'] );
		} else {
			$title = '';
			$address = 'Los Angeles CA';
			$key =	'';
			$width = '100%';
			$height = '250px';
			$zoom =	15;
			$enablescrollwheel = 'true';
			$draggable = 'true';
			$disablecontrols = 'false';
			$static = 'false';
			$style = 'roadmap';
			$show_address = 'false';
			$show_directions = 'false';
		}
		
		?>
	
		<!-- Widget Title -->
		<p>
			<label for="<?php echo $this->get_field_id('title'); ?>"><?php _e('Widget Title', 'bne-widgets'); ?></label> 
			<input class="widefat" id="<?php echo $this->get_field_id('title'); ?>" name="<?php echo $this->get_field_name('title'); ?>" type="text" value="<?php echo $title; ?>" />
		</p>

		<!-- API Key -->
		<p>
			<label for="<?php echo $this->get_field_id('key'); ?>"><?php _e('Google Maps API', 'bne-widgets'); ?>:</label>
			<span style="display:block;padding:2px 0" class="description"><?php echo _e( 'To display a Google map requires an active API key. <a href="https://developers.google.com/maps/documentation/javascript/get-api-key" target="_blank">Create a Key</a>.', 'bne-widgets'); ?></span>
			<input class="widefat" id="<?php echo $this->get_field_id('key'); ?>" name="<?php echo $this->get_field_name('key'); ?>" type="text" value="<?php echo $key; ?>" required/>
		</p>

		<!-- Address -->
		<p>
			<label for="<?php echo $this->get_field_id('address'); ?>"><?php _e('Address', 'bne-widgets'); ?>:</label>
			<span style="display:block;padding:2px 0" class="description"><?php echo _e( 'The address for the map location to be centered on.', 'bne-widgets'); ?></span>
			<input class="widefat" id="<?php echo $this->get_field_id('address'); ?>" name="<?php echo $this->get_field_name('address'); ?>" type="text" value="<?php echo $address; ?>" required/>
		</p>

		<!-- Embed Type -->
		<p>
			<label for="<?php echo $this->get_field_id('static'); ?>"><?php _e('Embed Type', 'bne-widgets'); ?>:</label>
			<span style="display:block;padding:2px 0" class="description"><?php echo _e( 'An interactive map will embed the map using javascript (similar to an iframe) allowing the user to use the below controls and interact with it. In contrast, a static map will be a pre-made image from Google\'s server of the address area.', 'bne-widgets'); ?></span>
			<select name="<?php echo $this->get_field_name('static'); ?>" id="<?php echo $this->get_field_id('static'); ?>" class="widefat">
				<?php
					echo '<option value="false" id="false"', $static == 'false' ? ' selected="selected"' : '', '>'.esc_html__( 'Interactive map', 'bne-widgets' ).'</option>';
					echo '<option value="true" id="true"', $static == 'true' ? ' selected="selected"' : '', '>'.esc_html__( 'Static map image', 'bne-widgets' ).'</option>';
				?>
			</select>
		</p>

		<!-- Style -->
		<p>
			<label for="<?php echo $this->get_field_id('style'); ?>"><?php _e('Map Style', 'bne-widgets'); ?>:</label>
			<select name="<?php echo $this->get_field_name('style'); ?>" id="<?php echo $this->get_field_id('style'); ?>" class="widefat">
				<?php
					echo '<option value="roadmap" id="roadmap"', $style == 'roadmap' ? ' selected="selected"' : '', '>'.esc_html__( 'Roadmap - This is the default map type.', 'bne-widgets' ).'</option>';
					echo '<option value="satellite" id="satellite"', $style == 'satellite' ? ' selected="selected"' : '', '>'.esc_html__( 'Satellite - Displays Google Earth satellite images', 'bne-widgets' ).'</option>';
					echo '<option value="hybrid" id="hybrid"', $style == 'hybrid' ? ' selected="selected"' : '', '>'.esc_html__( 'Hybrid -  Displays a mixture of normal and satellite views', 'bne-widgets' ).'</option>';
					echo '<option value="terrain" id="terrain"', $style == 'terrain' ? ' selected="selected"' : '', '>'.esc_html__( 'Terrain - Displays a physical map based on terrain information.', 'bne-widgets' ).'</option>';
				?>
			</select>
		</p>

		<!-- Width -->
		<p>
			<label for="<?php echo $this->get_field_id('width'); ?>"><?php _e('Map Width', 'bne-widgets'); ?>:</label>
			<span style="display:block;padding:2px 0" class="description"><?php echo _e( 'The width of the map. Include either %, rem, em, px, etc. If displaying as a static map image use px.', 'bne-widgets'); ?></span>
			<input class="widefat" id="<?php echo $this->get_field_id('width'); ?>" name="<?php echo $this->get_field_name('width'); ?>" type="text" value="<?php echo $width; ?>" />
		</p>

		<!-- Height -->
		<p>
			<label for="<?php echo $this->get_field_id('height'); ?>"><?php _e('Map Height', 'bne-widgets'); ?>:</label>
			<span style="display:block;padding:2px 0" class="description"><?php echo _e( 'The height of the map. Include either %, rem, em, px, etc. If displaying as a static map image use px.', 'bne-widgets'); ?></span>
			<input class="widefat" id="<?php echo $this->get_field_id('height'); ?>" name="<?php echo $this->get_field_name('height'); ?>" type="text" value="<?php echo $height; ?>" />
		</p>

		<!-- Zoom -->
		<p>
			<label for="<?php echo $this->get_field_id('zoom'); ?>"><?php _e('Zoom', 'bne-widgets'); ?>:</label>
			<span style="display:block;padding:2px 0" class="description"><?php echo _e( 'The zoom resolution or distance of the map (1 to 20). Higher the number the closer to the ground the position of the zoom will be.', 'bne-widgets'); ?></span>
			<input class="widefat" id="<?php echo $this->get_field_id('zoom'); ?>" name="<?php echo $this->get_field_name('zoom'); ?>" min="1" max="20" type="number" value="<?php echo $zoom; ?>" />
		</p>

		<!-- Scroll -->
		<p>
			<label for="<?php echo $this->get_field_id('enablescrollwheel'); ?>"><?php _e('Enable Scroll Wheel', 'bne-widgets'); ?>:</label>
			<span style="display:block;padding:2px 0" class="description"><?php echo _e( 'Interactive Map only. Allows the user to zoom in/out of the map with the mouse.', 'bne-widgets'); ?></span>
			<select name="<?php echo $this->get_field_name('enablescrollwheel'); ?>" id="<?php echo $this->get_field_id('enablescrollwheel'); ?>" class="widefat">
				<?php
					echo '<option value="true" id="true"', $enablescrollwheel == 'true' ? ' selected="selected"' : '', '>'.esc_html__( 'True', 'bne-widgets' ).'</option>';
					echo '<option value="false" id="false"', $enablescrollwheel == 'false' ? ' selected="selected"' : '', '>'.esc_html__( 'False', 'bne-widgets' ).'</option>';
				?>
			</select>
		</p>

		<!-- Draggable -->
		<p>
			<label for="<?php echo $this->get_field_id('draggable'); ?>"><?php _e('Draggable', 'bne-widgets'); ?>:</label>
			<span style="display:block;padding:2px 0" class="description"><?php echo _e( 'Interactive Map only. Allows the user to pan the map with the mouse.', 'bne-widgets'); ?></span>
			<select name="<?php echo $this->get_field_name('draggable'); ?>" id="<?php echo $this->get_field_id('draggable'); ?>" class="widefat">
				<?php
					echo '<option value="true" id="true"', $draggable == 'true' ? ' selected="selected"' : '', '>'.esc_html__( 'True', 'bne-widgets' ).'</option>';
					echo '<option value="false" id="false"', $draggable == 'false' ? ' selected="selected"' : '', '>'.esc_html__( 'False', 'bne-widgets' ).'</option>';
				?>
			</select>
		</p>

		<!-- Controls -->
		<p>
			<label for="<?php echo $this->get_field_id('disablecontrols'); ?>"><?php _e('Controls', 'bne-widgets'); ?>:</label>
			<span style="display:block;padding:2px 0" class="description"><?php echo _e( 'Interactive Map only. Displays the map controls as an overlay.', 'bne-widgets'); ?></span>
			<select name="<?php echo $this->get_field_name('disablecontrols'); ?>" id="<?php echo $this->get_field_id('disablecontrols'); ?>" class="widefat">
				<?php
					echo '<option value="true" id="true"', $disablecontrols == 'true' ? ' selected="selected"' : '', '>'.esc_html__( 'Hide Controls', 'bne-widgets' ).'</option>';
					echo '<option value="false" id="false"', $disablecontrols == 'false' ? ' selected="selected"' : '', '>'.esc_html__( 'Show Controls', 'bne-widgets' ).'</option>';
				?>
			</select>
		</p>

		<!-- Show Address -->
		<p>
			<label for="<?php echo $this->get_field_id('show_address'); ?>"><?php _e('Show Address', 'bne-widgets'); ?>:</label>
			<span style="display:block;padding:2px 0" class="description"><?php echo _e( 'Display the entered address below the map.', 'bne-widgets'); ?></span>
			<select name="<?php echo $this->get_field_name('show_address'); ?>" id="<?php echo $this->get_field_id('show_address'); ?>" class="widefat">
				<?php
					echo '<option value="true" id="true"', $show_address == 'true' ? ' selected="selected"' : '', '>'.esc_html__( 'True', 'bne-widgets' ).'</option>';
					echo '<option value="false" id="false"', $show_address == 'false' ? ' selected="selected"' : '', '>'.esc_html__( 'False', 'bne-widgets' ).'</option>';
				?>
			</select>
		</p>

		<!-- Show Directions -->
		<p>
			<label for="<?php echo $this->get_field_id('show_directions'); ?>"><?php _e('Show Direction Button', 'bne-widgets'); ?>:</label>
			<span style="display:block;padding:2px 0" class="description"><?php echo _e( 'Display a Directions button below the map.', 'bne-widgets'); ?></span>
			<select name="<?php echo $this->get_field_name('show_directions'); ?>" id="<?php echo $this->get_field_id('show_directions'); ?>" class="widefat">
				<?php
					echo '<option value="false" id="false"',$show_directions == 'false' ? ' selected="selected"' : '', '>'.esc_html__( 'False', 'bne-widgets' ).'</option>';
					echo '<option value="true" id="true"', $show_directions == 'true' ? ' selected="selected"' : '', '>'.esc_html__( 'True', 'bne-widgets' ).'</option>';
				?>
			</select>
		</p>




    <?php 
    
    } // End Form
     
}
$BNE_Widgets_Map = new BNE_Widgets_Map();