<?php
/*
 *	Form Support Functions
 *
 * 	@author		Kerry Kline
 * 	@link		http://www.bnecreative.com
 *	@package	BNE Testimonials Pro
 *
*/
	
// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit; 



/*
 *	Submission Form Fields
 *
 *	Fields are arranged using this array. Developers can tap into
 *	the array using bne_testimonials_form_fields to alter or add
 *	to the fields shown on the frontend.
 *
 *	@atts		array		options from [bne_testimonials_form]
 *	@returns	$fields
 *
 *	@since 		v2.4.2
 *	@updated	v2.8
 *
*/
function bne_testimonials_get_form_fields( $atts ) {
	
	// Get current user
	$current_user = wp_get_current_user();

	// Fields Array, Devs can filter 
	$fields = array(
		'name'	=>	array(
			'id'		=>	'name',
			'label'		=>	$atts['name_label'],
			'type'		=>	'text',
			'required'	=>	'true',
			'default'	=>	( $atts['auto_fill'] == 'true' && is_user_logged_in() ) ? $current_user->display_name : '',
		),
		'email'	=>	array(
			'id'		=>	'email',
			'label'		=>	$atts['email_label'],
			'type'		=>	'email',
			'required'	=>	'true',
			'default'	=>	( $atts['auto_fill'] == 'true' && is_user_logged_in() ) ? $current_user->user_email : '',
		),
		'message'	=>	array(
			'id'		=>	'message',
			'label'		=>	$atts['message_label'],
			'type'		=>	'textarea',
			'required'	=>	'true',
		),
		'thumbnail'	=>	array(
			'id'		=>	'thumbnail',
			'label'		=>	$atts['image_label'],
			'type'		=>	'file',
			'required'	=>	'false',
			'class'		=>	( $atts['image'] == 'false' ) ? 'hide' : '',
		),
		'category'	=>	array(
			'id'		=>	'category',
			'label'		=>	$atts['category_label'],
			'type'		=>	'category_select',
			'required'	=>	'false',
			'class'		=>	( $atts['category'] == 'false' ) ? 'hide' : '',
		),
		'tagline'	=>	array(
			'id'		=>	'tagline',
			'label'		=>	$atts['tagline_label'],
			'type'		=>	'text',
			'required'	=>	'false',
		),
		'website'	=>	array(
			'id'		=>	'website',
			'label'		=>	$atts['website_label'],
			'type'		=>	'text',
			'required'	=>	'false',
		),
		'rating'	=>	array(
			'id'		=>	'rating',
			'label'		=>	$atts['rating_label'],
			'type'		=>	'select',
			'required'	=>	'false',
			'options'	=>	array(
				'5'			=>	__( '5 Stars', 'bne-testimonials' ),
				'4'			=>	__( '4 Stars', 'bne-testimonials' ),
				'3'			=>	__( '3 Stars', 'bne-testimonials' ),
				'2'			=>	__( '2 Stars', 'bne-testimonials' ),
				'1'			=>	__( '1 Star', 'bne-testimonials' ),
				'0'			=>	__( 'No Rating', 'bne-testimonials' ),
			),
		),
		'terms_gdpr'		=>	array(
			'id'		=>	'terms_gdpr',
			'label'		=>	$atts['terms_label'],
			'type'		=>	'checkbox',
			'required'	=>	'true',
		),
	);

	// Star Rating Field
	if( 'stars' == $atts['rating'] ) {
		$fields['rating']['type'] = 'rating_star';
		unset( $fields['rating']['options']['0']);
	}
	

	// Show/hide fields based on atts
	$visiblity = array(
		'thumbnail'		=> 	$atts['image'],
		'tagline' 		=> 	$atts['tagline'],
		'website' 		=> 	$atts['website'],
		'rating'		=>	$atts['rating'],
		'terms_gdpr'	=>	$atts['terms'],
	);
	foreach( $visiblity as $k => $v ) {
		if( $v == 'false' ) {
			unset($fields[$k]);
		}
	};


	return apply_filters( 'bne_testimonials_form_fields', $fields, $atts );	
	
}



/*
 *	Form Processor
 *	Listens for when the form shortcode has been submitted
 *	and passes variables to the database.
 *
 *	@since 		v1.7
 *	@updated 	v2.9.13
 *
*/
function bne_testimoinals_form_submit() {
	
	if( isset( $_POST['bne_testimonials_nonce_field'] ) && wp_verify_nonce( $_POST['bne_testimonials_nonce_field'], 'bne_testimonials_nonce' ) ) {
		
		
		/*
		 *	Testimonial submit data
		 *	
		 *	Builds a data array of the submitted data for submission
		 *	to the database and create a new testimonial post.
		 *
		*/
		$form_data = array(
			'notify' 			=>	$_POST['notify'],
			'to_email' 			=>	$_POST['to_email'],
			'honey' 			=>	sanitize_text_field( $_POST['alternate_email'] ),
			'redirect_url'		=>	esc_url( $_POST['redirect_url'] ),
			'name' 				=>	sanitize_text_field( $_POST['testimonial_name'] ),
			'message' 			=>	sanitize_text_field( $_POST['testimonial_message'] ),
			'email' 			=>	sanitize_email( $_POST['testimonial_email'] ),
			'tagline' 			=>	isset( $_POST['testimonial_tagline'] ) ? sanitize_text_field( $_POST['testimonial_tagline'] ) : '',
			'website' 			=>	isset( $_POST['testimonial_website'] ) ? esc_url( $_POST['testimonial_website'] ) : '',
			'rating' 			=>	isset( $_POST['testimonial_rating'] ) ? esc_html( $_POST['testimonial_rating'] ) : '',
			'terms_gdpr' 		=>	isset( $_POST['terms_gdpr'] ) ? current_time( 'mysql' ).' - '.esc_html( $_POST['terms_gdpr'] ) : '',
		);
			
	
		/*
		 *	Spam Bot
		 *	
		 *	Checks if the the honeypot field has any data, if so
		 *	kill the submission!
		 *
		*/
		if( !empty( $form_data['honey'] ) ) {
			die( "Sorry, but you look to be a bot!" );
		}
		
		// Devs - Allow extending the pre submission process
		do_action( 'bne_testimoinals_form_submit_before', $form_data );
		

		/*
		 *	Required Field Checks
		 *
		 *	We only require three fields, the name (post title), email and the message.
		 *	An easy check is to see if there is at least data entered.
		 *
		 *	If pass, continue to creating the post.
		 *
		*/
		if( strlen( trim( $form_data['name'] ) ) < 1 || strlen( trim( $form_data['message'] ) ) < 1 ) {

			// Set Failed Redirect
			$redirect = esc_url_raw( add_query_arg( 'bne_testimonials_status', 'failed', get_permalink() ) );

		} else {


			/*
			 *	Create a new post!
			 *
			 *	Sets the basic info to creat a new testimonial post.
			 *	This allows us to then get the new ID to add the 
			 *	remaining data.
			 *
			*/
			$form_data['id'] = wp_insert_post( array(
				'post_type' 	=>	'bne_testimonials',
				'post_status' 	=>	apply_filters( 'bne_testimoinals_form_post_status', 'pending' ),
				'post_title' 	=>	$form_data['name'],
				'post_content' 	=>	$form_data['message'],
				'taxonomy'		=>	'bne-testimonials-taxonomy'
			) );



			/*
			 *	Now that a Testimonial ID has been created, we can send
			 *	the custom field data, featured image, admin notification,
			 *	and set the successful redirect.
			 *
			*/
			if( $form_data['id'] ) {
				
				
				// Set Categories
				if( isset( $_POST['cat'] ) ) {
					wp_set_post_terms( $form_data['id'], $_POST['cat'], 'bne-testimonials-taxonomy', false );
				}

				
				// Set Featured Image
			    if( isset( $_FILES['testimonial_thumbnail'] ) ) {
				    if( $_FILES['testimonial_thumbnail']['name'] ) {
					    $thumbnail = $_FILES['testimonial_thumbnail'];
					    if( $thumbnail ) {
						    bne_testimonials_upload_images( 'testimonial_thumbnail', $form_data['id'], true );
					    }
				    }
			    }


				// Set Tagline
				if( $form_data['tagline'] ) { 
					update_post_meta( $form_data['id'], 'tagline', $form_data['tagline'] );
				}


				// Set Email
				if( $form_data['email'] ) { 
					update_post_meta( $form_data['id'], 'email', $form_data['email'] );
				}


				// Set Website URL
				if( $form_data['website'] ) { 
					update_post_meta( $form_data['id'], 'website-url', $form_data['website'] );
				}


				// Set Rating
				if( $form_data['rating'] ) { 
					update_post_meta( $form_data['id'], 'rating', $form_data['rating'] );
				} else {
					update_post_meta( $form_data['id'], 'rating', '0' );
				}


				// Set Consent / Terms
				if( $form_data['terms_gdpr'] ) { 
					update_post_meta( $form_data['id'], 'terms_gdpr', $form_data['terms_gdpr'] );
				}


				/*
				 *	Set successful redirect
				 *
				 *	When successful, the page will redirect to the URL in the redirect_url.
				 *	If not specified, redirect to the same page and display the default
				 *	successful message.
				 *
				*/
				$redirect = esc_url_raw( add_query_arg( 'bne_testimonials_status', 'successful', $form_data['redirect_url'] ) );				

				

				/*
				 *	Initiate Admin Email Notification
				 *
				 *	If notify is true in the form, send the notification
				 *	email to the specified emai laddress with basic information
				 *	using bne_testimonials_email_notification()
				 *
				*/
				if( $form_data['notify'] == 'true' ) {
					bne_testimonials_email_notification( $form_data );
				}
				
				
				// Devs - Allow extending the final submission process
				do_action( 'bne_testimoinals_form_submit_after', $form_data );
				

			} // END Post Creating / Updating

		} // END Successful Posting

		wp_redirect( $redirect ); exit;
	}
}
add_action( 'init', 'bne_testimoinals_form_submit' );



/*
 *	Upload Media function
 *
 *	A wrapper to handle the uploading of the file to
 *	the WP Media Library
 *
 *	@since v2.4.2
*/
function bne_testimonials_upload_images( $file_handler, $post_id, $setthumb = false ) {
	
	// check to make sure its a successful upload
	if( $_FILES[$file_handler]['error'] !== UPLOAD_ERR_OK ) __return_false();

	// Make sure to include the WordPress media uploader API if it's not
	if( !function_exists( 'media_handle_upload' ) ) {
		require_once( ABSPATH . 'wp-admin/includes/image.php' );
		require_once( ABSPATH . 'wp-admin/includes/file.php' );
		require_once( ABSPATH . 'wp-admin/includes/media.php' );
	}

	$attach_id = media_handle_upload( $file_handler, $post_id );
 
	if( $setthumb ) {
		update_post_meta( $post_id, '_thumbnail_id', $attach_id );
	}
	
	return $attach_id;
	
}



/*
 *	Notification Email
 *
 *	By default, the notification email is active and
 *	will be sent to the main WP Admin Email address.
 *
 *	@since 		v1.7
 *	@updated 	v2.8.9
 *
*/
function bne_testimonials_email_notification( $form_data ) {
	
	// Get Options
	$options = get_option('bne_testimonials_settings');

	/*
	 *	Collect email data and allow developers to adjust it.
	 *	This is also used for the {field_list} dynamic variables.
	*/
	$email_data = apply_filters( 'bne_testimonials_email_data', array(
		esc_html__( 'From', 'bne-testimonials' ) 		=>	$form_data['name'],
		esc_html__( 'Email', 'bne-testimonials' ) 		=>	$form_data['email'],
		esc_html__( 'Tagline', 'bne-testimonials' ) 	=>	$form_data['tagline'],
		esc_html__( 'Website', 'bne-testimonials' ) 	=>	$form_data['website'],
		esc_html__( 'Rating', 'bne-testimonials' ) 		=>	$form_data['rating'],
		esc_html__( 'Message', 'bne-testimonials' ) 	=>	$form_data['message'],
		esc_html__( 'Terms', 'bne-testimonials' ) 		=>	$form_data['terms_gdpr'],
	) );

	// Loop through $email_data and build an unordered list for the email.
	$email_data_imploded = '<ul style="padding: 0; margin: 0; list-style: none;">';
	foreach( $email_data as $label => $value ) {
		if( $value ) {
			$email_data_imploded .= '<li><strong>'.$label.':</strong> '.$value.'</li>';
		}
	}
	$email_data_imploded .= '</ul>';
	

	// Convert Dynamic Variables from settings
	$dynamic_vars = array(
		'{field_list}'	=>	$email_data_imploded,
		'{name}'		=>	$form_data['name'],
		'{email}'		=>	$form_data['email'],
		'{tagline}'		=>	$form_data['tagline'],
		'{website}'		=>	$form_data['website'],
		'{rating}'		=>	$form_data['rating'],
		'{message}'		=>	$form_data['message'],
		'{terms}'		=>	$form_data['terms_gdpr'],
	);



	// To
	$to = $form_data['to_email'];
	
	// From (admin)
	$from = ( isset( $options['form_email'] ) ) ?  $options['form_email'] : get_bloginfo( 'admin_email' );
	$from = apply_filters( 'bne_testimonials_wp_from_email', $from );

	// Subject
	$subject = ( isset( $options['form_subject'] ) ) ? $options['form_subject'] : esc_html__( 'You have a new Testimonial', 'bne-testimonials' );
	
	// Message
	if( isset( $options['form_message'] ) ) {
		$message = wpautop( strtr( $options['form_message'], $dynamic_vars ) );

	// Default (legacy message)
	} else {
		$message = '<p>'.sprintf( 
			__( 'You have received a new testimonial at %s from the below person. This testimonial is not published yet and needs to be approved.', 'bne-testimonials' ),
			'<a href="'.esc_url( get_bloginfo( 'url' ) ).'">'.get_bloginfo( 'name' ).'</a>'
		).'</p>';

		$message .= $email_data_imploded;
	
	}
	
	
	
	// TEST DATA
	/*
	$message .= '<hr>';
	foreach( $_POST as $label => $value) {
	    if(strpos( $label, 'testimonial_') === 0 ) {
		    $label = str_replace( array( 'testimonial_', '-', '_' ), ' ', $label );
	        $message .= '<strong>'.ucwords($label).'</strong>: '.$value.'<br>';
	    }
	}
	*/
	
	
	// Headers
	$headers = array(
		'From: '.$from,
		'Reply-To: '.$form_data['email'],
		'Content-Type: text/html'
	);

	// Send It!
	wp_mail( $to, $subject, $message, $headers );
			
}