<?php
/*
 *	Register Mian Testimonials Custom Post Type
 *
 * 	@author		Kerry Kline
 * 	@link		http://www.bnecreative.com
 *	@package	BNE Testimonials Pro
 *	@since		v2.3
 *
*/

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit; 


class BNE_Testimonials_Pro_CPT {


	/*
	 *	Post type name
	 *
	 *	@var string $post_type_name Holds the name of the post type.
	 *
	*/
	public $post_type_name = 'bne_testimonials';
	

    /*
     * 	Constructor
     *
     *	@since v2.0
     *	@updated v2.5
     *
    */
	function __construct() {

		
	    /*
	     * 	Hook into BNE_CPT and register a new custom post type.
	     *	Translation is taken care of in the BNE_CPT class.
	     *
	    */
		$cpt = new BNE_CPT( $this->post_type_name, array(
			'singular'		=> 	esc_html__( 'Testimonial', 'bne-testimonials' ),
			'plural'		=> 	esc_html__( 'Testimonials', 'bne-testimonials' ),
			'slug' 			=> 	'testimonials',
			'args'			=>	array(
				'labels'		=>	array(
					'menu_name' 	=> 	esc_html__( 'Testimonials', 'bne-testimonials' ),
				),
				'supports'				=>	array( 'title', 'editor', 'thumbnail' ),
				'menu_icon'				=>	'dashicons-id-alt',
			    'public'             	=> 	false,
			    'publicly_queryable' 	=> 	false,
			    'exclude_from_search'	=>	true,
			    'show_ui'            	=> 	true,
			    'show_in_menu'       	=> 	true,
			    'show_in_nav_menus'  	=> 	false,
			    'show_in_admin_bar'  	=> 	false,
			    'hierarchical'       	=> 	false,
			    'has_archive'        	=> 	false,
			),
		));
		

	    /*
	     * 	Hook into BNE_CPT's register_taxonomy function
	     *	and register a new custom taxonomy to our custom
	     *	post type.
	     *
	     *	Translation is taken care of in the BNE_CPT class.
	     *
	    */
		$cpt->register_taxonomy( 'bne-testimonials-taxonomy', array(
			'singular' 	=> 	esc_html__( 'Category', 'bne-testimonials' ),
			'plural'	=> 	esc_html__( 'Categories', 'bne-testimonials' ),
			'slug'		=>	'categories',
			'args'		=> 	array(
				'labels'	=>	array(
					'menu_name'		=>	esc_html__( 'Categories', 'bne-testimonials' ),
				),
				'show_admin_column' 	=> 	true,
			    'hierarchical'       	=> 	true,
			    'public'             	=> 	false,
			    'show_ui'            	=> 	true,
			    'show_tagcloud'      	=> 	false,
			    'show_in_nav_menus'  	=> 	false,
			    'show_admin_column'  	=> 	true,
			    'rewrite'            	=>	false,
			)
		));


		
	    /*
	     * 	Hook into BNE_CPT's set_taxomony function
	     *	and set our translation slug.
	     *
	    */
		$cpt->set_textdomain( 'bne-testimonials' );


	    // Post List Columns
		add_filter( "manage_edit-{$this->post_type_name}_columns", array( $this, 'columns' ), 10, 1 );
		add_action( "manage_{$this->post_type_name}_posts_custom_column",  array( $this, 'column_content' ), 10, 2 );
		
		// Menu Notification
		add_filter( 'add_menu_classes', array( $this, 'menu_bubble' ) );
		
		// Post Admin
		add_filter( 'enter_title_here', array( $this, 'post_title' ) );
		add_filter( 'admin_post_thumbnail_html', array( $this, 'post_thumbnail_html' ) );
		
		// CPT CMB2 Fields
		add_action( 'cmb2_admin_init', array( $this, 'cmb2_metabox' ) );
		add_action( 'cmb2_before_post_form_bne_testimonials_details_metabox', array( $this, 'cmb2_metabox_scripts' ), 10, 2 );
		
	}




	/*
	 *	Add Admin Menu Bubble
	 *
	 *	@since		v1.7
	 *	@updated 	v2.3
	 *
	*/
	function menu_bubble( $menu ) {
	
	  	$count_posts = wp_count_posts( 'bne_testimonials' );
	  	$pending_posts = $count_posts->pending;
	
	    foreach( $menu as $menu_key => $menu_data ) {
	        if( 'edit.php?post_type='.$this->post_type_name != $menu_data[2] ) {
	            continue;
	        }
	        $menu[$menu_key][0] .= ' <span class="update-plugins count-'.$pending_posts.'"><span class="plugin-count">'.number_format_i18n( $pending_posts ).'</span></span>';
	    }
	    
	    return $menu;
	}
	



	/*
	 *	Add/Remove Columns on Post List Admin Screen
	 *
	 *	@since 		v1.1
	 * 	@updated 	v2.5
	 *
	*/
	function columns( $columns ) {
		
		// Remove Columns
		unset( $columns['date'] );
	
		// Add Columns
	    $columns['title'] 				= esc_html__( 'Name', 'bne-testimonials' );
	    $columns['testimonial_message'] = esc_html__( 'Message', 'bne-testimonials' );
	    $columns['testimonial_rating'] 	= esc_html__( 'Rating', 'bne-testimonials' );
	    $columns['testimonial_image'] 	= esc_html__( 'Image', 'bne-testimonials' );
	    $columns['date'] 				= esc_html__( 'Published', 'bne-testimonials' );

	    return $columns;
	
	}




	/*
	 *	Add Content to Columns on Post List Admin Screen
	 *
	 *	@since 		v1.1
	 * 	@updated 	v2.3
	 *
	*/
	function column_content( $column, $post_id ) {
	
		// Message
		if( $column === 'testimonial_message' ) {
			echo substr( get_the_excerpt(), 0, 60 ) . '...';
	    }
	
		// Rating
		if( $column === 'testimonial_rating' ) {
		 
			$rating = get_post_meta( $post_id, 'rating', true );
			$empty = ($rating > 0 ) ? 5 - $rating : '5';
			
			// Rating Gold Stars
			for( $stars = 1; $stars <= $rating; $stars++ ) {
				echo '<span class="dashicons dashicons-star-filled" style="color:gold;"></span>';
			}
			
			// Rating Empty Stars or Not rated
			for( $stars = 1; $stars <= $empty; $stars++ ) {
				echo '<span class="dashicons dashicons-star-empty" style="color:#ccc;"></span>';
			}
		
		}
	
		// Image
		if( $column === 'testimonial_image' ) {
			echo the_post_thumbnail( array( 80, 80 ) );
	    }	
	
	}




	/*
	 *	Post Title Label
	 *
	 *	@since 		v1.1
	 *	@updated 	v2.3
	 *
	*/
	function post_title( $title ) {
		$screen = get_current_screen();
		
		if( $this->post_type_name == $screen->post_type ) {
			$title = bne_testimonials_get_local( 'cpt_post_title' );
		}
		
		return $title;
	}
	



	/*
	 *	Featured Image Widget Text
	 *
	 *	@since 		v1.1
	 *	@updated 	v2.3
	 *
	*/
	function post_thumbnail_html( $content ) {
		global $post_type;
		if( $this->post_type_name == $post_type ) {
			$content .= bne_testimonials_get_local( 'cpt_thumb_description' );
		}
	    return $content;
	}
	

	/*
	 *	Meta Box
	 *
	 *	Defines a set of meta boxes for our CPT.
	 *
	 *	@since 		v1.0
	 *	@updated 	v2.7.3
	 *
	*/
	function cmb2_metabox() {
	
		// Set field prefix
		$prefix = $this->post_type_name.'_';
	
	
		// Initiate Metabox
		$cmb = new_cmb2_box( array(
			'id'            =>	'bne_testimonials_details_metabox',
			'title'         =>  bne_testimonials_get_local( 'cpt_details_title' ),
			'object_types'  =>  array( $this->post_type_name ),
			'classes'		=>	'bne-cmb-wrapper'
		) );
	
		// Tagline
		$cmb->add_field( array(
			'name'    			=>	bne_testimonials_get_local( 'cpt_tagline_title' ),
			'desc'    			=>	bne_testimonials_get_local( 'cpt_tagline_description'),
			'id'      			=>	'tagline',
			'type'    			=>	'text',
			'sanitization_cb' 	=> 	array( $this, 'sanitize_tagline' ),
			'column'			=>	true,
			'classes'			=>	'has-input-wrap'
		) );

		
		// Website URL
		$cmb->add_field( array(
			'name'    		=>	bne_testimonials_get_local( 'cpt_website_title' ),
			'desc'    		=>	bne_testimonials_get_local( 'cpt_website_description' ),
			'id'      		=>	'website-url',
			'type'    		=>	'text_url',
			'before_field'	=>	'<div class="bne-input-prepend"><span class="dashicons dashicons-admin-site"></span></div>',
			'attributes'	=>	array(
				'placeholder'	=>	'http://www.google.com/',
			),
			'classes'		=>	'has-input-wrap'
		) );


		// Email
		$cmb->add_field( array(
			'name'    		=>	bne_testimonials_get_local( 'cpt_email_title' ),
			'desc'    		=>	bne_testimonials_get_local( 'cpt_email_description'),
			'id'      		=>	'email',
			'type'    		=>	'text_email',
			'before_field'	=>	'<div class="bne-input-prepend"><span class="dashicons dashicons-email"></span></div>',
			'classes'		=>	'has-input-wrap'
		) );


		// Rating
		$cmb->add_field( array(
			'name'    		=>	bne_testimonials_get_local( 'cpt_rating_title' ),
			'desc'    		=>	bne_testimonials_get_local( 'cpt_rating_description' ),
			'id'      		=>	'rating',
			'type'    		=>	'select',
			'default'		=>	5,
			'options'		=>	array(
				5				=>	esc_html__( '5 Stars', 'bne-testimonials' ),	
				4				=>	esc_html__( '4 Stars', 'bne-testimonials' ),	
				3				=>	esc_html__( '3 Stars', 'bne-testimonials' ),	
				2				=>	esc_html__( '2 Stars', 'bne-testimonials' ),	
				1				=>	esc_html__( '1 Star', 'bne-testimonials' ),	
				0				=>	esc_html__( 'No Rating', 'bne-testimonials' ),	
			),
		) );

		// Consent / Terms
		$cmb->add_field( array(
			'name'    		=>	bne_testimonials_get_local( 'cpt_terms_title' ),
			'desc'    		=>	bne_testimonials_get_local( 'cpt_terms_description' ),
			'id'      		=>	'terms_gdpr',
			'type'    		=>	'text',
			'attributes'	=>	array(
				'readonly'		=>	true
			),
			'before_field'	=>	'<div class="bne-input-prepend"><span class="dashicons dashicons-lock"></span></div>',
			'classes'		=>	'has-input-wrap'
		) );
		
	}
	

	
	/*
	 *	Testimonial Text field Sanitize Callback
	 *	Allows minor html tags within the tagline textfield.
	 *
	 *	@value	string	The value entered in the field before saving.
	 *
	 *	@note	http://php.net/manual/en/function.strip-tags.php
	 *	@since 		v2.3.1
	 *	@updated 	v2.5
	 *
	*/
	function sanitize_tagline( $value ) {
	    return strip_tags( $value, '<i><b><br><strong>' );
	}



	/*
	 *	Meta Box CSS / JS
	 *
	 *	Injects needed scripts/styles for our fields
	 *
	 *	@since 	v2.3
	 *
	*/
	function cmb2_metabox_scripts( $post_id, $cmb ) {
		?>
		<!-- Admin Form JS -->
		<script type="text/javascript">
			jQuery(document).ready(function($) { 
				$('.bne-cmb-wrapper .has-input-wrap input').wrap('<div class="bne-input-wrap"></div>');
			});
		</script>	
	
		<?php
	}
	
} // END Class

$BNE_Testimonials_Pro_CPT = new BNE_Testimonials_Pro_CPT();