<?php
/*
 * 	Init class for BNE Slider
 *
 * 	@author		Kerry Kline
 * 	@copyright	Copyright (c) Kerry Kline
 * 	@link		http://www.bnecreative.com
 * 	@package 	BNE WordPress Framework
 *	@package	BNE Slider
 *
*/

// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;


if( !class_exists( 'BNE_Slider' ) ) {
	class BNE_Slider {
		
		function __construct() {

			// Requirement Notice and dependencies
			add_action( 'admin_init', function() {
				$pass = true;
				if( !defined( 'BNE_FRAMEWORK_VERSION' ) || ( defined( 'BNE_FRAMEWORK_VERSION' ) && BNE_FRAMEWORK_VERSION < '2' ) ) {
					$pass = false;
				}
				if( !defined( 'CMB2_LOADED' ) ) {
					$pass = false;
				}
				
				if( false == $pass ) {	
					add_action( 'admin_notices', function() {
						echo '<div class="notice notice-error is-dismissible"><p>Apologies, but <strong>BNE Slider</strong> requires a theme running BNE Theme Framework v2.0+</p></div>';
					});
					deactivate_plugins( BNE_SLIDER_BASENAME );
				}
				
			});
			
			// Setup Hook
			add_action( 'after_setup_theme', array( $this, 'setup' ) );

		}
		
		/*
		 * 	Plugin Setup / Hooks
		 *
		*/
		function setup() {
			
			// Scripts
			add_action( 'wp_enqueue_scripts', array( $this, 'scripts' ) );

			// Shortcodes
			add_shortcode( 'slider', array( $this, 'shortcode' ) );
			
			// CPT Hooks
			add_filter( 'manage_edit-bne_slider_columns', array( $this, 'columns' ), 10, 1 );
			add_action( 'manage_bne_slider_posts_custom_column', array( $this, 'column_content' ), 10, 2 );
			add_action( 'cmb2_admin_init', array( $this, 'metabox' ) );
			add_action( 'cmb2_before_post_form_bne_slides_slider_meta_box', array( $this, 'metabox_scripts' ), 10, 2 );

			// BNE Slider Crop
			add_image_size( 'slider-full', 1680, 560, true );
			
			//Hook into BNE_CPT and register a new custom post type.
			if( class_exists( 'BNE_CPT' ) ) {
				$cpt = new BNE_CPT( 'bne_slider', array(
					'singular'					=> 	esc_html__( 'Slider', 'bne-slider' ),
					'plural'					=> 	esc_html__( 'Sliders', 'bne-slider' ),
					'args'						=>	array(
						'labels'					=>	array(
							'menu_name' 				=>	esc_html__( 'Sliders', 'bne-slider' ),
						),
						'supports'				=>	array( 'title' ),
						'menu_icon'				=>	'dashicons-images-alt2',
					    'public'             	=> 	false,
					    'publicly_queryable' 	=> 	false,
					    'show_ui'            	=> 	true,
					    'show_in_menu'       	=> 	true,
					    'show_in_nav_menus'  	=> 	false,
					    'show_in_admin_bar'  	=> 	false,
					    'hierarchical'       	=> 	false,
					    'has_archive'        	=> 	false,
					),
				));					
			}


			/*
			 *	Plugin Update Function (BNE Server)
			 *	Allows updating the plugin from the WP Dashboard.
			 *
			*/
			require( BNE_SLIDER_DIR .'/lib/update-checker/plugin-update-checker.php' );
			$BNE_Slider_update = Puc_v4_Factory::buildUpdateChecker(
				'https://updates.bnecreative.com/?action=get_metadata&slug=bne-slider',
				dirname( __FILE__ ).'/bne-slider.php',
				'bne-slider'
			);
			
			
		}

				
		/*
		 * 	Register plugin scripts/styles
		 *
		*/
		function scripts() {
			$min = ( defined('WP_DEBUG') && true === WP_DEBUG ) ? '' : '.min';
			wp_register_script( 'flexslider', BNE_SLIDER_URI . '/assets/jquery.flexslider-min.js', array('jquery'), '2.7.2', true );
			wp_register_style( 'bne_slider', BNE_SLIDER_URI . '/assets/bne-slider'.$min.'.css', '', BNE_SLIDER_VERSION, 'all' );
		}

			
		/*
		 * 	[slider] Shortcode
		 *	Accepts: id, shadow, crop
		 *
		*/
		function shortcode( $atts ) {
			$atts = shortcode_atts( array(
				'id'				=> 	'',
				'shadow'			=> 	'',
				'crop' 				=> 	'slider-full',
				'smooth_height'		=> 	'true',
				'play_pause'		=> 	'false',
				'animation_speed'	=> 	'700',
				'mobile_image'		=>	'',
				'image_loading'		=>	'lazy',
				'class'				=>	''
		
			), $atts, 'slider' );
		
			// Setup Query
			$query_args = array(
				'post_type' 		=> 'bne_slider',
				'posts_per_page'	=> 1,
				'name'				=> $atts['id']
			);
		
			
			// Enqueue Scripts
			wp_enqueue_style( 'bne_slider' );
			wp_enqueue_script( 'flexslider' );
						
			
			// Empty
			$output = '';
		
			// Start Query
			$bne_slider = new WP_Query( $query_args );
			if( $bne_slider->have_posts() ) {
		
				// The Loop
				while ( $bne_slider->have_posts() ) : $bne_slider->the_post();
		
					// Get Slider Options
					$animation 	= get_post_meta( get_the_id(), 'bne_slider_transition', true );
					$autoplay 	= get_post_meta( get_the_id(), 'bne_slider_autoplay', true );
					$speed 		= esc_html( get_post_meta( get_the_id(), 'bne_slider_duration', true ) );
					$nav 		= get_post_meta( get_the_id(), 'bne_slider_nav', true );
					$arrows 	= get_post_meta( get_the_id(), 'bne_slider_arrows', true );
					$pause 		= get_post_meta( get_the_id(), 'bne_slider_pause', true );
					$slides 	= get_post_meta( get_the_ID(), 'bne_slider_slides', true );
					$mobile_image_id = get_post_meta( get_the_ID(), 'bne_slider_mobile_image_id', true );
					if( $mobile_image_id || $atts['mobile_image'] ) {
						$atts['class'] .= ' has-mobile-image';
					}
					
					// Set shadow
					if( !$atts['shadow'] ) {
						$atts['shadow'] = get_post_meta( get_The_ID(), 'bne_slider_shadow', true );
					}
		
					// Load Flexslider API
					wp_add_inline_script( 'flexslider', 
						'jQuery(document).ready(function($){
							$("#bne-slider-'.$atts['id'].'.bne-flexslider").flexslider({
								useCSS: false,
								video: true,
								prevText: \'<i class="bne-fa fas fa-angle-left"></i><span class="sr-only">Previous Slide</span>\',
								nextText: \'<i class="bne-fa fas fa-angle-right"></i><span class="sr-only">Next Slide</span>\',
								pauseText: \'<i class="bne-fa fas fa-pause"></i>\',
								playText: \'<i class="bne-fa fas fa-play"></i>\',
								animation:     "'.$animation.'",
								slideshow: '.( $autoplay ? $autoplay : 'true').',
								slideshowSpeed: '.$speed.',
								animationSpeed: '.$atts['animation_speed'].',
								controlNav:   	'.$nav.',
								directionNav: 	'.$arrows.',
								pauseOnHover: 	'.$pause.',
								smoothHeight: 	'.$atts['smooth_height'].',
								controlsContainer: "#bne-slider-'.$atts['id'].' .bne-flexnav-container",
								pausePlay: '.$atts['play_pause'].',
						
								start: function(slider) {
									$(slider).find(".flex-active-slide .animate").addClass("visible");
								},
								before: function(slider) {
									$(slider).find(".flex-active-slide .animate").each(function(){
										$(this).removeClass("visible");
									});
								},
								after: function(slider) {
									$(slider).find(".flex-active-slide .animate").addClass("visible");
								}
							}).parent().find(".bne-loader").fadeOut();
						});'
					);
		
		
					// Build Slider Wrapper
					$output .= '<div class="bne-slider-wrapper crop-'.$atts['crop'].' '.$atts['class'].' clearfix">';
					
						// Mobile Image
						if( $atts['mobile_image'] ) {
							$output .= '<img src='.$atts['mobile_image'].' alt="" class="slider-mobile-image" style="display:none;/>';
						} elseif( $mobile_image_id ) {
							$output .= wp_get_attachment_image($mobile_image_id, 'full', null, array( 'class'=>'slider-mobile-image', 'style' => 'display:none;' ) );
						} 

						$output .= '<div class="slides-inner">';
									
							// Build Flexslider
							$output .= '<div id="bne-slider-'.$atts['id'].'" class="bne-flexslider '.$animation.'-transition">';
								
								$output .= '<div class="bne-loader"><i class="bne-fa fas fa-spinner fa-spin"></i></div>';
								
								$output .= '<ul class="slides" style="list-style: none;">';
		
									// loop through slide data
									foreach( $slides as $key => $slide ) {
										$option = array(
											'image'			=>	isset( $slide['image_slide_id'] ) ? $slide['image_slide_id'] : '',
											'link'			=>	isset( $slide['image_link'] ) ? esc_url( $slide['image_link'] ) : '',
											'shade'			=>	isset( $slide['shade'] ) ? $slide['shade'] : '',
											'headline'		=>	isset( $slide['headline'] ) ? wpautop( do_shortcode( $slide['headline'] ) ) : '',
											'description'	=>	isset( $slide['description'] ) ? wpautop( do_shortcode( $slide['description'] ) ) : '',
											'color'			=>	isset( $slide['caption_color'] ) ? $slide['caption_color'] : 'text-light',
											'placement'		=>	isset( $slide['caption_placement'] ) ? $slide['caption_placement'] : 'center_center'
										);
										
										// Build Slide
										$output .= '<li class="slide">';
											$output .= '<div class="slide-inner-content">';
												
												// Open Slide Link
												if( $option['link'] ) {
													$output .= '<a href="'.$option['link'].'">';
												}
												
											
												// BG Shade
												if( $option['shade'] ) {
													$output .= '<div class="bg-shade" style="display: block;background-color:'.$option['shade'].';"></div>';
												}
		
												// Image Slide
												if( $option['image'] ) {
		
													// Linked Image
													if( $option['link'] ) {
														$output .= wp_get_attachment_image( $option['image'], $atts['crop'], '', array( 'class'	=> 'bne-slider-link', 'loading' => $atts['image_loading'] ) );
														// Close Link
														$output .= '</a>';
													} else {
														$output .= wp_get_attachment_image( $option['image'], $atts['crop'], '', array( 'loading' => $atts['image_loading'] ) );
													}
												
													// Caption Overlay
													if( $option['headline'] || $option['description'] ) {
														$output .= '<div class="slide-caption '.$option['placement'].'">';
		
															if( $option['headline'] ) {
																$output .= '<div class="slider-headline '.$option['color'].' ">'.$option['headline'].'</div>';
															}
		
															if( $option['description'] ) {
																$output .= '<div class="slider-description '.$option['color'].'">'.$option['description'].'</div>';
															}
		
														$output .= '</div><!-- .slide-caption (end) -->';
													}
		
												}
		
											$output .= '</div><!-- .slide-inner-content (end) -->';
		
										$output .= '</li><!-- .slide (end) -->';
									
									} // END Foreach
									
		
								$output .= '</ul><!-- .slides (end) -->';
		
								$output .= '<div class="bne-flexnav-container"></div>';
		
							$output .= '</div><!-- .bne-flexslider (end) -->';
						$output .= '</div><!-- .slides-inner (end) -->';
					$output .= '</div><!-- .bne-slider-wrapper (end) -->';
		
					// Add shadow to slider frame
					if( $atts['shadow'] ) {
						$output .= '<div class="slider-shadow shadow-type'.$atts['shadow'].'"></div>';
					}
		
				endwhile;
		
			// No slides to display
			} else {
				$output .= '<div class="alert alert-warning"><strong>Slider:</strong> "'.$atts['id'].'". '.esc_html__( 'This slider ID does not exist.', 'bne' ).'</div>';
			}
		
			// Reset Query
			wp_reset_postdata();
		
			// Return data
			return $output;
		
		} // END Shortcode


	
		/*
		 *	Add/Remove Columns on Post List Admin Screen
		 *
		 * 	@since 		v1.0
		 *	@updated 	v1.3.2
		 *
		*/
		function columns( $columns ) {
		
			unset( $columns['date'] );
			unset( $columns['author'] );
		
		    $columns['title'] 			= esc_html__( 'Name', 'bne' );
		    $columns['slide_images'] 	= esc_html__( 'Slide Images', 'bne' );
		    $columns['shortcode'] 		= esc_html__( 'Slider Shortcode', 'bne' );
		
		    return $columns;
		}
		
	
		/*
		 *	Add Content to Columns on Post List Admin Screen
		 *
		 * 	@since 		v1.0
		 *	@updated 	v1.3.7
		 *
		*/
		function column_content( $column, $post_id ) {
		
			global $post;
		
			// Slide Images
			if( $column === 'slide_images') {
				$slides = get_post_meta( $post_id, 'bne_slider_slides', true );
				foreach( $slides as $key => $slide ) {
					$image = isset( $slide['image_slide_id'] ) ? $slide['image_slide_id'] : '';
					if( $image ) {
						echo '<span style="display: inline-block; margin: 5px;">'.wp_get_attachment_image( $image, array(60,60) ).'</span>';
					}
				}
			}
		
			// Slider Shortcode
			if( $column === 'shortcode' ) {
				echo '[slider id="'.$post->post_name.'"]';
			}
		
		}
		
	
	
		/*
		 *	Meta Box
		 *
		 *	Defines a set of meta boxes for our CPT.
		 *
		*/
		function metabox() {
	
			// Set field prefix
			$prefix = 'bne_slider_';
		
			// Register CMB2 Metabox
			$slides_metabox = new_cmb2_box( array(
				'id'           	=> 	'bne_slides_slider_meta_box',
				'title'        	=> 	esc_html__( 'Available Slides', 'bne' ),
				'object_types' 	=> 	array( 'bne_slider', ),
				'classes'		=>	'bne-cmb-wrapper',
			) );

			// Slide Group
			$slides_group_id = $slides_metabox->add_field( array(
				'id'          => $prefix . 'slides',
				'type'        => 'group',
				'description' => '',
				'options'     => array(
					'group_title'   => 	esc_html__( 'Slide {#}', 'bne' ),
					'add_button'    => 	esc_html__( 'Add Another Slide', 'bne' ),
					'remove_button' => 	esc_html__( 'Remove Slide', 'bne' ),
					'sortable'      => 	true,
				),
			) );
		
			// Image
			$slides_metabox->add_group_field( $slides_group_id, array(
				'name'    		=>	esc_html__( 'Set Image', 'bne' ),
				'desc'    		=>	esc_html__( 'By default, images are hard cropped to 1680x560 unless you specify the crop option within the shortcode.', 'bne' ),
				'id'      		=>	'image_slide',
				'type'    		=>	'file',
				'options' 		=>	array(
					'url'      				=> 	false,
					'add_upload_file_text' 	=>  esc_html__( 'Select Image', 'bne' ),
				),
				'preview_size' 	=> 	'slider-full',
			) );
		
			// Image Link
			$slides_metabox->add_group_field( $slides_group_id, array(
				'name'				=>	esc_html__( 'Image Link', 'bne' ),
				'desc'				=>	esc_html__( 'If you wish to have the image link to another page, add a full URL here.', 'bne' ),
				'id'				=>  'image_link',
				'type'				=>  'text',
				'attributes'		=> 	array(
					'placeholder'		=>	'http://www.website.com/page/',
				),
				'before_field'	=>	'<div class="bne-input-prepend">URL</div>',
				'classes'		=>	'has-input-wrap',
			) );
		
			// BG Shade
			$slides_metabox->add_group_field( $slides_group_id, array(
				'name'			=>	esc_html__( 'Shade Color and Opacity', 'bne' ),
				'desc'			=>	esc_html__( "Add a shaded background over this slide to darken or lighten it's appearance.", 'bne' ),
				'id'			=>  'shade',
				'type'			=>  'colorpicker',
				'default'		=>	'',
				'options' 		=>	array(
					'alpha' 		=>	true,
				)
			) );
			
			// Headline
			$slides_metabox->add_group_field( $slides_group_id, array(
				'name'			=>	esc_html__( 'Headline', 'bne' ),
				'desc'			=>	esc_html__( 'Add an optional Headline that will display over the image.', 'bne' ),
				'id'			=>  'headline',
				'type'			=>  'textarea',
				'attributes'	=>	array(
					'rows'			=> '3',
				)
			) );
		
			// Description
			$slides_metabox->add_group_field( $slides_group_id, array(
				'name'			=>	esc_html__( 'Description', 'bne' ),
				'desc'			=>	esc_html__( 'Add an optional description that will display over the image. Basic HTML and shortcodes can be used.', 'bne' ),
				'id'			=>  'description',
				'type'			=>  'textarea',
				'attributes'	=>	array(
					'rows'			=> '6',
				)
			) );
		
			// Text Color
			$slides_metabox->add_group_field( $slides_group_id, array(
				'name'			=>	esc_html__( 'Text Color', 'bne' ),
				'desc'			=>	esc_html__( 'Force the headline and message to either be dark or light text based on the slide image used.', 'bne' ),
				'id'			=>  'caption_color',
				'type'			=>  'select',
				'options'		=>	array(
					'text-default'	=>	esc_html__( 'Default color text', 'bne' ),
					'text-light'	=>	esc_html__( 'Light color text', 'bne' ),
					'text-dark'		=>	esc_html__( 'Dark color text', 'bne' ),
				)
			) );
		
			// Text Placement
			$slides_metabox->add_group_field( $slides_group_id, array(
				'name'			=>	esc_html__( 'Text Placement', 'bne' ),
				'desc'			=>	esc_html__( 'The location of the headline and description text over the slide image.', 'bne' ),
				'id'			=>  'caption_placement',
				'type'			=>  'select',
				'default'		=>	'center_center',
				'options'		=>	array(
					'left_top'			=>	esc_html__( 'Left Top', 'bne' ),
					'left_center'		=>	esc_html__( 'Left Center', 'bne' ),
					'left_bottom'		=>	esc_html__( 'Left Bottom', 'bne' ),
					'center_top'		=>	esc_html__( 'Center Top', 'bne' ),
					'center_center'		=>	esc_html__( 'Center Center', 'bne' ),
					'center_bottom'		=>	esc_html__( 'Center Bottom', 'bne' ),
					'right_top'			=>	esc_html__( 'Right Top', 'bne' ),
					'right_center'		=>	esc_html__( 'Right Center', 'bne' ),
					'right_bottom'		=>	esc_html__( 'Right Bottom', 'bne' ),
				)
			) );
			// ====== END GROUP FIELDS ====== //
		
		
	
			// Register CMB2 Metabox
			$slider_settings = new_cmb2_box( array(
				'id'           	=> 	'bne_slides_slider_settings_meta_box',
				'title'        	=> 	esc_html__( 'Slider Options', 'bne' ),
				'object_types' 	=> 	array( 'bne_slider', ),
				'context'   	=> 	'side',
				'priority'   	=> 	'default',
				'classes'		=>	'bne-cmb-wrapper',
			) );
		
			// Transition
			$slider_settings->add_field( array(
				'name'    			=>	esc_html__( 'How to transition between slides?', 'bne' ),
				'desc'    			=>	'',
				'id'      			=>	$prefix . 'transition',
			    'type'             	=> 	'select',
			    'show_option_none' 	=> 	false,
			    'options'          	=> 	array(
					'fade' 				=>	'Fade',
					'slide' 			=> 	'Slide',
				),
				'attributes' 		=> 	array(
					'style'				=>	'width:98%;'
				),
			) );
		
			// Duration
			$slider_settings->add_field( array(
			    'name'     			=>	esc_html__( 'Seconds between each transition?', 'bne' ),
			    'desc'     			=> 	esc_html__( 'Hint: 5000 milliseconds = 5 seconds', 'bne' ),
				'id'      			=>	$prefix . 'duration',
			    'type'     			=> 	'text',
				'default'			=> '5000',
				'attributes' 		=> 	array(
					'type' 				=> 	'number',
					'pattern' 			=> 	'\d*',
					'style'				=>	'width:98%;'
				),
			) );
		
			// Nav Pagination
			$slider_settings->add_field( array(
				'name'    			=>	esc_html__( 'Show slideshow navigation?', 'bne' ),
				'desc'    			=>	'',
				'id'      			=>	$prefix . 'nav',
			    'type'             	=> 	'select',
			    'show_option_none' 	=> 	false,
			    'options'          	=> 	array(
					'true' 				=> 	esc_html__( 'Yes, show the navigation.', 'bne' ),
					'false' 			=> 	esc_html__( "No, don't show it.", 'bne' ),
				),
				'attributes' 		=> 	array(
					'style'				=>	'width:98%;'
				),
			) );
		
			// Arrows
			$slider_settings->add_field( array(
				'name'    			=>	esc_html__( 'Show next/prev arrows?', 'bne' ),
				'desc'    			=>	'',
				'id'      			=>	$prefix . 'arrows',
			    'type'             	=> 	'select',
			    'show_option_none' 	=> 	false,
			    'options'          	=> 	array(
					'true' 				=> 	esc_html__( 'Yes, show the arrows.', 'bne' ),
					'false' 			=> 	esc_html__( "No, don't show it.", 'bne' ),
				),
				'attributes' 		=> 	array(
					'style'				=>	'width:98%;'
				),
			) );
		
			// Pause on Hover
			$slider_settings->add_field( array(
				'name'    			=>	esc_html__( 'Enable pause on hover?', 'bne' ),
				'desc'    			=>	'',
				'id'      			=>	$prefix . 'pause',
			    'type'             	=> 	'select',
			    'show_option_none' 	=> 	false,
			    'options'          	=> 	array(
					'false' 			=>  esc_html__( 'No, continuously play the slider.', 'bne' ),
					'true' 				=>  esc_html__( 'Pause when mouse hovers.', 'bne' ),
				),
				'attributes' 		=> 	array(
					'style'				=>	'width:98%;'
				),
			) );

			// Autoplay
			$slider_settings->add_field( array(
				'name'    			=>	esc_html__( 'Autoplay Slideshow?', 'bne' ),
				'desc'    			=>	'',
				'id'      			=>	$prefix . 'autoplay',
			    'type'             	=> 	'select',
			    'show_option_none' 	=> 	false,
			    'options'          	=> 	array(
					'true' 				=>  esc_html__( 'Yes, begin the slideshow automatically.', 'bne' ),
					'false' 			=>  esc_html__( 'No, require user to control the slideshow.', 'bne' ),
				),
				'attributes' 		=> 	array(
					'style'				=>	'width:98%;'
				),
			) );

			// Shadow
			$slider_settings->add_field( array(
				'name'    			=>	esc_html__( 'Shadow (bottom of slider frame)', 'bne' ),
				'desc'    			=>	'',
				'id'      			=>	$prefix . 'shadow',
			    'type'             	=> 	'select',
			    'show_option_none' 	=> 	false,
			    'options'          	=> 	array(
					'0' 				=>  esc_html__( 'No Shadow', 'bne' ),
					'1' 				=>  esc_html__( 'Shadow Style #1', 'bne' ),
					'2' 				=>  esc_html__( 'Shadow Style #2', 'bne' ),
					'3' 				=>  esc_html__( 'Shadow Style #3', 'bne' ),
					'4' 				=>  esc_html__( 'Shadow Style #4', 'bne' ),
					'5' 				=>  esc_html__( 'Shadow Style #5', 'bne' ),
					'6' 				=>  esc_html__( 'Shadow Style #6', 'bne' ),
					'7' 				=>  esc_html__( 'Shadow Style #7', 'bne' ),
					'8' 				=>  esc_html__( 'Shadow Style #8', 'bne' ),
				),
				'attributes' 		=> 	array(
					'style'				=>	'width:98%;'
				),
			) );

			// Image
			$slider_settings->add_field( array(
				'name'    		=>	esc_html__( 'Mobile Image', 'bne' ),
				'desc'    		=>	esc_html__( 'If set, this image will be used in-place of the slider on mobile screens.', 'bne' ),
				'id'      		=>	$prefix . 'mobile_image',
				'type'    		=>	'file',
				'options' 		=>	array(
					'url'      				=> 	false,
					'add_upload_file_text' 	=>  esc_html__( 'Select Image', 'bne' ),
				),
				'preview_size' 	=> 	'slider-full',
			) );
		
		}
		
		
		/*
		 *	Inject JS/CSS for the form on page.
		 *
		 *	Instead of enqueing up another file on the admin,
		 *	we are just going to add it to the form itself.
		 *
		*/
		function metabox_scripts( $post_id, $cmb ) {
			
			?>		
			<!-- Slider Meta Box CSS -->
			<style type="text/css">
				#minor-publishing { display: none; }
				#bne_slides_slider_meta_box .handlediv,
				#bne_slides_slider_meta_box .hndle { display: none; }
				#bne_slides_slider_meta_box { border: none; background: none; box-shadow: none; }
				
				.bne-cmb-wrapper .cmb-group-title { background: white !important; }
				.bne-cmb-wrapper .cmb-repeat-group-wrap,
				.bne-cmb-wrapper .cmb-repeat-group-wrap > .cmb-td { background: none !important; }
				.bne-cmb-wrapper .cmb-repeat-group-wrap > .cmb-td { padding-left: 0 !important; padding-right: 0 !important; }
				
				.cmb2-wrap.bne-cmb-wrapper .cmb2-metabox .cmb-repeatable-group > .cmb-row:last-of-type,
				.cmb2-wrap.bne-cmb-wrapper .cmb2-metabox .cmb-repeatable-group > .cmb-row:last-of-type .cmb-td { background: none !important;}
				
				.cmb-group-description { margin-bottom: 15px !important; margin-top: -25px !important; }
				.cmb-group-description .cmb2-metabox-description { color: #666; font-weight: normal; }
			</style>
			<?php
		}
	
	}
}