<?php
/*
 * Plugin Name: BNE Scripts
 * Version: 1.0
 * Plugin URI: http://www.bnecreative.com
 * Description:  Adds additional options to include custom scripts and tags in the theme header and footer for BNE themes.
 * Author: Kerry Kline
 * Author URI: http://www.bnecreative.com
 * Requires at least: 5.0
 * License: GPL2

    Copyright 2019  BNE Creative

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License version 2,
    as published by the Free Software Foundation.

    You may NOT assume that you can use any other version of the GPL.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    The license for this software can likely be found here:
    http://www.gnu.org/licenses/gpl-2.0.html

*/


// Exit if accessed directly
if ( !defined('ABSPATH')) exit;


// INIT Class
class BNE_Scripts {

	
	/*
 	 * 	Option key used in the Database
 	 *	 
 	 * 	@var string
 	*/
	private $options_key = 'bne_scripts_options';
	
	
	/*
 	 * 	Metabox ID assigned using CMB2
 	 *	 
 	 * 	@var string
 	*/
	private $metabox_id = 'bne_scripts_option_metabox';


    /*
     * 	Constructor
     *
     *	@since v1.0
     *
    */
	function __construct() {
		
		// Set Constants
		define( 'BNE_SCRIPTS_VERSION', '1.0' );
		define( 'BNE_SCRIPTS_DIR', dirname( __FILE__ ) );
		define( 'BNE_SCRIPTS_URI', plugins_url( '', __FILE__ ) );
		define( 'BNE_SCRIPTS_BASENAME', plugin_basename( __FILE__ ) );
		
		// Requirement Notice and dependencies
		add_action( 'admin_init', function() {
			$pass = true;
			if( !defined( 'BNE_FRAMEWORK_VERSION' ) ) {
				$pass = false;
			}
			if( !defined( 'CMB2_LOADED' ) ) {
				$pass = false;
			}
			
			if( false == $pass ) {	
				add_action( 'admin_notices', function() {
					echo '<div class="notice notice-error is-dismissible"><p>Apologies, but <strong>BNE Scripts</strong> requires a theme running BNE Theme Framework</p></div>';
				});
				deactivate_plugins( BNE_SCRIPTS_BASENAME );
			}
			
		});

		// Register the Metabox
		add_action( 'cmb2_admin_init', array( $this, 'register_settings' ) );
		
		// Hook before Metabox
		add_action( "cmb2_before_options-page_form_{$this->metabox_id}", array( $this, 'before_metabox' ), 10, 2 );
		
		// Hook After Metabox
		add_action( "cmb2_after_options-page_form_{$this->metabox_id}", array( $this, 'after_metabox' ), 10, 2 );

		// Output
		add_action( 'wp_head', array( $this, 'header_scripts' ) );
		add_action( 'wp_footer', array( $this, 'footer_scripts' ) );
	}



	/*
	 * 	Register the Settings Metabox and Settings Page
	 *	Hooks into CMB2 to automatically register the page and options metabox
	 *
	*/
	function register_settings() {

		/**
		 * Registers options page menu item and form.
		 */
		$cmb_options = new_cmb2_box( array(
			'id'				=> 	$this->metabox_id,
			'title'				=> 	esc_html__( 'Header & Footer Scripts', 'bne-scripts' ),
			'menu_title'		=>	esc_html__( 'Scripts', 'bne-scripts' ),
			'object_types'		=> 	array( 'options-page' ),
			'option_key'		=> 	$this->options_key,
			'parent_slug'		=> 	'options-general.php',
			'capability'		=> 	'manage_options',
		) );
		

		// == Section: Analytics == //
		$cmb_options->add_field( array(
			'name' 				=> 	esc_html__( 'Analytics', 'bne-scripts' ),
			'desc' 				=> 	esc_html__( 'Include tracking scripts on the frontend for analytic data.', 'bne-scripts' ),
			'id'   				=> 	'section_analytics',
			'type' 				=> 	'title',
		) );

		// GA Code
		$cmb_options->add_field( array(
		    'name'      		=> 	esc_html__( 'Google Tracking ID', 'bne-scripts' ),
		    'desc'      		=> 	esc_html__( 'Input your Google Analytics "Tracking ID".', 'bne-scripts' ),
		    'id'        		=> 	'ga_id',
		    'type'     			=> 	'text',
		    'placeholder'		=>	'Ex: UA-12345678-9',
		) );

		// GA Anonymize IP
		$cmb_options->add_field( array(
		    'name'      		=> 	esc_html__( 'IP Anonymization', 'bne-scripts' ),
		    'desc'      		=> 	esc_html__( 'Anonymize IP addresses in tracking code.', 'bne-scripts' ),
		    'id'        		=> 	'ga_anonymize',
		    'type'     			=> 	'checkbox',
		) );


		// == Section: Scripts == //
		$cmb_options->add_field( array(
			'name' 				=> 	esc_html__( 'Custom Scripts and Tags', 'bne-scripts' ),
			'desc' 				=> 	esc_html__( 'Easily add custom scripts, meta tags, tracking and analytic codes, within the header or footer of your website. Anything added here will be included on every frontend page.', 'bne-scripts' ),
			'id'   				=> 	'section_scripts',
			'type' 				=> 	'title',
		) );

		// Header Scripts
		$cmb_options->add_field( array(
			'name' 				=> 	__('Header Scripts', 'scripts'),
			'id' 				=> 	'header_scripts',
			'type' 				=> 	'textarea',
			'attributes'		=>	array(
				'style'				=>	'width: 99%',
			),
			'sanitization_cb'	=>	false,
		) );
	
		// Footer Scripts
		$cmb_options->add_field( array(
			'name' 				=> 	__('Footer Scripts', 'scripts'),
			'id' 				=> 	'footer_scripts',
			'type' 				=> 	'textarea',
			'attributes'		=>	array(
				'style'				=>	'width: 99%',
			),
			'sanitization_cb'	=>	false,
		) );

	}
	

	/*
	 * 	Before Metabox Hook
	 *	Adds content before the CMB2 metabox
	 *
	*/
	function before_metabox( $cmb_id, $object_id ) {
		echo '<div class="bne-cmb-options-wrapper">';
	}
	

	/*
	 * 	After Metabox Hook
	 *	Adds content after the CMB2 metabox
	 *
	*/
	function after_metabox( $cmb_id, $object_id ) {
		?>
		</div><!-- .bne-cmb-options-wrapper (end) -->
		<?php
	}
	

	/*
	 * 	Header Scripts Output
	 *
	*/
	function header_scripts() {
		$scripts = get_option( $this->options_key );
		
		if( !empty( $scripts['header_scripts'] ) ) {
			echo '<!-- Header Scripts -->';
			echo $scripts['header_scripts'], "\n";
			echo '<!-- /Header Scripts -->';
		}

		if( !empty( $scripts['ga_id'] ) ) {
			?>
			<!-- Google Analytics -->
			<script>
				(function(i,s,o,g,r,a,m){i['GoogleAnalyticsObject']=r;i[r]=i[r]||function(){
				(i[r].q=i[r].q||[]).push(arguments)},i[r].l=1*new Date();a=s.createElement(o),
				m=s.getElementsByTagName(o)[0];a.async=1;a.src=g;m.parentNode.insertBefore(a,m)
				})(window,document,'script','https://www.google-analytics.com/analytics.js','ga');
				ga('create', '<?php echo esc_attr( $scripts['ga_id'] ); ?>', 'auto');
				<?php if( !empty( $scripts['ga_anonymize'] ) ) { ?>
				ga('set', 'anonymizeIp', true);
				<?php } ?>
				ga('send', 'pageview');
			</script>
			<!-- /Google Analytics -->
			<?php
		}

	} // END header_scripts



	/*
	 * 	Footer Scripts Output
	 *
	*/
	function footer_scripts() {
		$scripts = get_option( $this->options_key );
		if( !empty( $scripts['footer_scripts'] ) ) {
			echo '<!-- Footer Scripts -->';
			echo $scripts['footer_scripts'], "\n";
			echo '<!-- /Footer Scripts -->';
		}
	} // END footer_scripts


} // END Class
	
// Initiate the Class
$BNE_Scripts = new BNE_Scripts();