<?php
/*
 * Plugin Name: BNE Navbar
 * Version: 1.3.4
 * Plugin URI: http://www.bnecreative.com
 * Description:  Adds a custom navigation toolbar to the bottom of your WordPress website allowing quick links and additional hidden menus. Once activated, go to Admin Menu > Settings > BNE Navbar to get started.
 * Author: Kerry Kline
 * Author URI: http://www.bnecreative.com
 * Text Domain: bne-navbar
 * Requires at least: 4.7
 * License: GPL2

    Copyright 2017-2018  BNE Creative

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License version 2,
    as published by the Free Software Foundation.

    You may NOT assume that you can use any other version of the GPL.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    The license for this software can likely be found here:
    http://www.gnu.org/licenses/gpl-2.0.html

*/


// Exit if accessed directly
if ( !defined('ABSPATH')) exit;


// INIT Class
class BNE_NavBar {

	
    /*
     * 	Constructor
     *
     *	@since v1.0
     *
    */
	function __construct() {
		
		// Set Constants
		define( 'BNE_NAVBAR_VERSION', '1.3.4' );
		define( 'BNE_NAVBAR_DIR', dirname( __FILE__ ) );
		define( 'BNE_NAVBAR_URI', plugins_url( '', __FILE__ ) );
		define( 'BNE_NAVBAR_BASENAME', plugin_basename( __FILE__ ) );
		define( 'BNE_NAVBAR_UPDATE_ID', 'bne-navbar' );
		
		// Textdomain
		add_action( 'plugins_loaded', array( $this, 'text_domain' ) );

		// Setup Includes / Files
		add_action( 'after_setup_theme', array( $this, 'setup' ) );
		
		// Scripts 
		add_action( 'wp_enqueue_scripts', array( $this, 'frontend_scripts' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ), 11, 1 );
		
	}




	/*
	 * 	Textdomain for Localization
	 *
	 * 	@since	v1.0
	*/
	function text_domain() {
		load_plugin_textdomain( 'bne-navbar', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
	}




	/*
	 *	Plugin Setup
	 *	
	 * 	@since 		v1.0
	 *	@updated	v1.3.2
	 *
	*/
	function setup() {

		// Admin Only Pages
		if( is_admin() ) {
			
			// CMB
			include_once( BNE_NAVBAR_DIR . '/includes/lib/cmb2/init.php' );

			// Settings Page
			include_once( BNE_NAVBAR_DIR . '/includes/settings.php' );
		
		// Frontend
		} else {
			add_action( 'wp_footer', array( $this, 'frontend_output' ) );
		}

		/*
		 *	Plugin Update Function (BNE Server)
		 *	Allows updating the plugin from the WP Dashboard.
		 *
		*/
		require( BNE_NAVBAR_DIR .'/includes/lib/update-checker/plugin-update-checker.php' );
		$BNE_Navbar_Update = Puc_v4_Factory::buildUpdateChecker(
			'https://updates.bnecreative.com/?action=get_metadata&slug='.BNE_NAVBAR_UPDATE_ID,
			__FILE__,
			BNE_NAVBAR_UPDATE_ID
		);
	
	}




	/*
	 *	Register frontend CSS and JS
	 *
	 *	@since 		v1.0
	 *	@updated	v1.3.2
	 *
	*/
	function frontend_scripts() {
	
		// Register
		wp_register_style( 'bne_navbar', BNE_NAVBAR_URI . '/assets/css/bne-navbar.css', '', BNE_NAVBAR_VERSION, 'all' );
		wp_register_script( 'bne_navbar', BNE_NAVBAR_URI . '/assets/js/bne-navbar.js', array('jquery'), BNE_NAVBAR_VERSION, true );

		// FontAwesome
		$options = get_option('bne_navbar_options');
		// Skip if disabled or using BNE Framework
		if( !defined('BNE_FRAMEWORK_VERSION') && isset( $options['include_fa'] ) == 'on' ) {
			// Allow Devs to use FontAwesome v5
			if( apply_filters( 'bne_navbar_fa_v5', false ) ) {
				wp_enqueue_script( 'fontawesome', '//use.fontawesome.com/releases/v5.0.8/js/all.js', '', '5.0.8', true );
				wp_enqueue_script( 'fontawesome-shims', '//use.fontawesome.com/releases/v5.0.8/js/v4-shims.js', '', '5.0.8', true );
			} else {
				wp_enqueue_style( 'fontawesome', BNE_NAVBAR_URI . '/assets/fontawesome/css/font-awesome.min.css', '', '4.7.0' );
			}
		}
		
		// Enqueue
		wp_enqueue_style( 'bne_navbar' );
		wp_enqueue_script( 'bne_navbar' );
	
	}

	
	
	/*
	 *	
	 *	Register Admin CSS and JS
	 *
	 *	@since 	v1.0
	 *
	*/
	function admin_scripts( $hook ) {
		if( $hook == 'settings_page_bne_navbar_options' ) {
			wp_enqueue_style( 'bne_navbar_admin', BNE_NAVBAR_URI . '/assets/css/bne-navbar-admin.css', '', BNE_NAVBAR_VERSION, 'all'  );
			wp_enqueue_script( 'bne_navbar_admin', BNE_NAVBAR_URI . '/assets/js/bne-navbar-admin.js', array('jquery'), BNE_NAVBAR_VERSION, true );
		}
	}

	


	/*
	 *
	 *	Frontend Output
	 *
	 *	@since 		v1.0
	 *	@updated	v1.3
	 *
	*/
	function frontend_output() {
		
		// Bail if options are not set yet!
		if( !get_option( 'bne_navbar_options' ) ) {
			return;
		}
		
		// Main Defaults
		$option_defaults = array(
			'nav_visibility'	=>	'hide_disabled',
			'nav_bg_color'		=>	'#ffffff',
			'nav_title_color'	=>	'#777777',
			'nav_icon_color'	=>	'#777777',
			'nav_buttons'		=>	array(),
			'nav_overlay'		=>	'off',
		);
		
		// Nav Button Defaults
		$nav_btn_defaults = array(
			'link'					=>	'#',
			'link_target'			=>	'off',
			'title'					=>	'',
			'icon'					=>	'',
			'image'					=>	'',
			'link_type'				=>	'page',
			'custom_menu_bg_color'	=>	'#222222',
			'custom_menu_heading'	=>	'',
			'custom_menu'			=>	'',
			'class'					=>	'',
		);
		
		
		// Parse Main Options
		$options = wp_parse_args( get_option( 'bne_navbar_options' ), $option_defaults );
				
		// Get number of columns
		$columns = count( $options['nav_buttons'] );
			
		// Classes
		$classes = ( is_array( $options['nav_visibility'] ) ) ? implode(' ', $options['nav_visibility'] ) : 'hide_disabled';
		$classes .= ' columns-'.$columns;
		$classes .= ' nav_bottom';
		
		// Nav ID
		$i = 0;
		
		// Open Nav Bar Wrapper
		echo '<div id="bne-navbar" class="active '.$classes.'" style="background-color:'.$options['nav_bg_color'].';">';
			
			// Overlay
			if( $options['nav_overlay'] == 'on' ) {
				echo '<div class="bne-navbar-overlay"></div>';
			}
			
			// Dev - Allow custom content above the nav links.
			do_action( 'bne_navbar_before' );
			
			echo '<div class="bne-navbar-inner">';
										
					// Loop
					foreach( $options['nav_buttons'] as $nav_button ) {
						
						$i++;

						
						// Parse Defaults
						$nav_button = wp_parse_args( $nav_button, $nav_btn_defaults );

						
						// Link Attributes
						$target = ( $nav_button['link_target'] == 'on' ) ? '_blank' : '';

						
						// Btn Classes
						$btn_classes = 'bne-navbar-link';

						// Hijack button link if it should go to a custom menu instead
						if( $nav_button['link_type'] == 'custom_menu' ) {
							$nav_button['link'] = '#bne-navbar-menu-'.$i;
							$btn_classes .= ' has-custom-menu';
						}
						
						// Open Button
						echo '<div id="bne-navbar-btn-'.$i.'" class="bne-navbar-btn-wrapper '.$nav_button['class'].'">';
						
							echo '<a href="'.esc_url( $nav_button['link'] ).'" target="'.$target.'" title="'.esc_html( $nav_button['title'] ).'" data-navlink="'.$i.'" class="'.$btn_classes.'">';
							
								// Icon
								$icon_prefix = apply_filters( 'bne_navbar_fa_prefix', 'fa-' );
								if( $nav_button['icon'] ) {
									$nav_button['icon'] = str_replace( 'fa-', '', $nav_button['icon'] );
	
									echo '<i class="bne-navbar-icon bne-fa fa '.esc_html( $icon_prefix.$nav_button['icon'] ).'" style="color: '.$options['nav_icon_color'].'"></i>';
								}
								
								// Image
								elseif( $nav_button['image'] ) {
									echo '<img src="'.esc_url( $nav_button['image'] ).'" class="bne-navbar-image" alt="'.esc_html( $nav_button['title'] ).'" />';
								}							
	
								// Title
								if( $nav_button['title'] ) {
									echo '<span class="bne-navbar-title" style="color: '.$options['nav_title_color'].'">'.esc_html( $nav_button['title'] ).'</span>';
								}						
	
							echo '</a>';
							
							// Custom Menu
							if( $nav_button['link_type'] == 'custom_menu' ) {
								
								echo '<div id="bne-navbar-menu-'.$i.'" class="bne-navbar-menu text-'.$nav_button['custom_menu_link_color'].'" style="background:'.$nav_button['custom_menu_bg_color'].';">';
									
									echo '<div class="bne-navbar-menu-inner">';
	
										// Heading
										echo '<div class="bne-navbar-menu-header">';
											echo '<span class="bne-navbar-title">'.esc_html( $nav_button['custom_menu_heading'] ).'</span>';
											echo '<div class="bne-navbar-menu-close"><i class="bne-fa fa '.$icon_prefix.'close"></i></div>';
										echo '</div>';	
										
										// Menu
										echo '<div class="bne-navbar-menu-body">';
										
											// Dev - Allow custom content before
											do_action( 'bne_navbar_menu_before', $i );
										
											if( $nav_button['custom_menu'] ) {
												wp_nav_menu( array( 
													'menu' 				=> $nav_button['custom_menu'],
													'menu_class' 		=> 'menu',
													'echo' 				=> true,
													'container_class' 	=> 'bne-navbar-custom-menu'
												) );
											} else {
												if( is_user_logged_in( 'manage_options' ) ) {
													echo '<p class="navbar-notice">'.__( 'Admin Notice: A custom Menu has not been selected for this nav button.', 'bne-navbar' ).'</p>';
												}
											}
											
											// Dev - Allow custom content below
											do_action( 'bne_navbar_menu_after', $i );
										
										echo '</div><!-- .bne-navbar-menu-body (end) -->';
	
									echo '</div><!-- .bne-navbar-menu-inner (end) -->';
										
								echo '</div><!-- .bne-navbar-menu (end) -->';
							
							} // END Custom Menu
					
						// Close Button
						echo '</div><!-- .bne-navbar-btn-wrapper (end) -->';

					} // END Loop
				
			echo '</div><!-- .bne-navbar-inner (end) -->';

			// Dev - Allow custom content after the nav links.
			do_action( 'bne_navbar_after' );

		echo '</div><!-- #bne-navbar (end) -->';

	}
	

} // END Class

	
// Initiate the Class
$BNE_NavBar = new BNE_NavBar();